/*
** sstest.c -- test/demonstration program for switchScanner
**

[BEGIN NOTICE]

Copyright 2005 Larry Hastings

This software is provided 'as-is', without any express or implied warranty.
In no event will the authors be held liable for any damages arising from
the use of this software.

Permission is granted to anyone to use this software for any purpose,
including commercial applications, and to alter it and redistribute
it freely, subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not
   claim that you wrote the original software. If you use this software
   in a product, an acknowledgment in the product documentation would be
   appreciated but is not required.
2. Altered source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.
3. This notice may not be removed or altered from any source distribution.
The switchScanner homepage is here:
	http://www.midwinter.com/~lch/programming/switchScanner/

[END NOTICE]
*/

#include <stdarg.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>

#include "scanner.h"


void basetest(token (*scanner)(char **s), char *s, va_list *list) {
	token value;
	char *commaString = "";

	printf("\"%s\" ->", s);

	do {
		int wanted = va_arg(*list, int);
		value = scanner(&s);
		if (value != wanted) {
			printf("\nTOKEN DIDN'T MATCH, WANTED %s (%d) GOT %s (%d)\n", scannerTokenLookup(wanted), wanted, scannerTokenLookup(value), value);
			exit(-1);
		}
		printf("%s %s (%d)", commaString, scannerTokenLookup(value), value);
		commaString = ",";
	} while (value > TOKEN_NOERROR);
	printf("\n");
}

void test(char *s, ...) {
	va_list list;
	va_start(list, s);
	basetest(scanner, s, &list);
	va_end(list);
}

void testUseScanner(token (*scanner)(char **s), char *s, ...) {
	va_list list;
	va_start(list, s);
	basetest(scanner, s, &list);
	va_end(list);
}


int main(void) {
	printf("success:\n");
	test("cat", TOKEN_CAT, TOKEN_EOF);
	test("bar", TOKEN_BAR, TOKEN_EOF);
	test("cdr", TOKEN_CDR, TOKEN_EOF);
	test("car", TOKEN_CAR, TOKEN_EOF);
	test("cart", TOKEN_CART, TOKEN_EOF);
	test(":", TOKEN_COLON, TOKEN_EOF);
	test(";", TOKEN_SEMICOLON, TOKEN_EOF);
	test("=", TOKEN_EQUALS, TOKEN_EOF);
	test("cart:car bar=cdr;cat", TOKEN_CART, TOKEN_COLON, TOKEN_CAR, TOKEN_BAR, TOKEN_EQUALS, TOKEN_CDR, TOKEN_SEMICOLON, TOKEN_CAT, TOKEN_EOF);

	testUseScanner(semicolon, " ; ;; ; ", TOKEN_SEMICOLON, TOKEN_SEMICOLON, TOKEN_SEMICOLON, TOKEN_SEMICOLON, TOKEN_EOF);
	testUseScanner(lettere, " e \t e \n e ", TOKEN_E, TOKEN_E, TOKEN_E, TOKEN_EOF);
	testUseScanner(sopwith, " sopwith \t sopwith  sopwith ", TOKEN_SOPWITH, TOKEN_SOPWITH, TOKEN_SOPWITH, TOKEN_EOF);
	testUseScanner(byandby, " byhook \t bycrook  byhook ", TOKEN_BYHOOK, TOKEN_BYCROOK, TOKEN_BYHOOK, TOKEN_EOF);

	printf("\nfailure:\n");
	test("c", TOKEN_ERROR);
	test("ca", TOKEN_ERROR);
	test("catt", TOKEN_ERROR);
	test("b", TOKEN_ERROR);
	test("ba", TOKEN_ERROR);
	test("crt", TOKEN_ERROR);
	test("balrog", TOKEN_ERROR);
	test("carta", TOKEN_ERROR);
	test("zaire", TOKEN_ERROR);

	printf("\npartial, fails:\n");
	test("car:zaire", TOKEN_CAR, TOKEN_COLON, TOKEN_ERROR);
	test("bar zaire:bar", TOKEN_BAR, TOKEN_ERROR);

	printf("\n\nF U CN RD THS, ALL TESTS PASSED.\n\n");


	{
	clock_t start;
	clock_t end;
	int i;
	int iterations = 5 * 1000 * 1000;

	start = clock();
	for (i = 0; i < iterations; i++)
		{
		char *t;
#define TWICE(x) t = x; scanner(&t);
		TWICE("cat");
		TWICE("bar");
		TWICE("cdr");
		TWICE("car");
		TWICE("cart");
		TWICE(":");
		TWICE(";");
		TWICE("=");

		t = "cart:car bar=cdr;cat";
		scanner(&t); scanner(&t); scanner(&t);
		scanner(&t); scanner(&t); scanner(&t);
		scanner(&t); scanner(&t);
		}
	end = clock();
	printf("%f seconds for %d tokens.\n", (end - start) / (float)CLOCKS_PER_SEC, iterations * 16); /* 16 tokens per iteration */
	printf("%f tokens per second.\n", ((double)iterations * 16 * CLOCKS_PER_SEC) / (end - start));
	}
	return 0;
}
