/*
** [BEGIN NOTICE]
**
** Nocog Copyright (C) 2005 Larry Hastings
**
** This software is provided 'as-is', without any express or implied warranty.
** In no event will the authors be held liable for any damages arising from
** the use of this software.
**
** Permission is granted to anyone to use this software for any purpose,
** including commercial applications, and to alter it and redistribute
** it freely, subject to the following restrictions:
**
** 1. The origin of this software must not be misrepresented; you must not
**    claim that you wrote the original software. If you use this software
**    in a product, an acknowledgment in the product documentation would be
**    appreciated but is not required.
** 2. Altered source versions must be plainly marked as such, and must not be
**    misrepresented as being the original software.
** 3. This notice may not be removed or altered from any source distribution.
**
** The Nocog homepage is here:
**		http://www.midwinter.com/~lch/programming/nocog/
**
** [END NOTICE]
*/



#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>



/*
** all of these macros assume the following
**  o  there is a variable "returnValue" of an integer/enum type
**  o  there a goto label in the current function called "EXIT"
**  o  zero is success, nonzero is failure
*/

#define RETURN(expr)            \
    {                           \
    returnValue = expr;         \
    goto EXIT;                  \
    }                           \
	
/* if the expression "expr" is nonzero, return whatever "expr" was */
#define ASSERT_SUCCESS(expr)    \
    {                           \
    returnValue = expr;         \
    if (returnValue)            \
	goto EXIT;              \
    }                           \
	
/* if the expression "expr" is false, return rv */
#define ASSERT_RETURN(expr, rv) \
    {                           \
    if (!(expr))                \
        {                       \
        returnValue = rv;       \
        goto EXIT;              \
        }                       \
    }                           \


#define COG_MARKER "[" "[" "[" "c" "o" "g"
char marker[1024];

enum unStatus 
	{
	unSuccess = 0,
	unFileError = 1,
	unStatError = 2,
	unMemoryError = 3,
	unNocogError = 4,

	unFourByteTrick = 0x10000000
	};



#ifdef _WIN32

#ifndef _DEBUG
	#pragma comment(linker, "/OPT:NOWIN98")
#endif // !_DEBUG


#include <windows.h>

unStatus fileGetContents(char *filename, char **bufferOut)
	{
	unStatus returnValue;
	char *buffer = NULL;

	HANDLE hFile;
	hFile = CreateFile(filename, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_ALWAYS, 0, NULL);
	ASSERT_RETURN(hFile != INVALID_HANDLE_VALUE, unFileError);

	DWORD size;
	size = GetFileSize(hFile, NULL);
	ASSERT_RETURN(size != INVALID_FILE_SIZE, unStatError);

	buffer = (char *)malloc(size);
	ASSERT_RETURN(buffer != NULL, unMemoryError);

	DWORD got;
	got = 0;
	ASSERT_RETURN(ReadFile(hFile, buffer, size, &got, NULL) != 0, unFileError);
	ASSERT_RETURN(got == size, unFileError);
	RETURN(unSuccess);

EXIT:
	if (returnValue == unSuccess)
		{
		*bufferOut = buffer;
		}
	else
		{
		if (buffer != NULL)
			free(buffer);
		}

	if (hFile != NULL)
		CloseHandle(hFile);
	
	return returnValue;
	}

#else // !_WIN32

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#define stricmp strcasecmp

unStatus fileGetContents(char *filename, char **bufferOut)
	{
	unStatus returnValue;
	char *buffer = NULL;
	
	int hFile;
	hFile = open(filename, O_RDONLY);
	ASSERT_RETURN(hFile != -1, unFileError);

	struct stat statBuffer;
	memset(&statBuffer, 0, sizeof(statBuffer));
	ASSERT_RETURN(stat(filename, &statBuffer) == 0, unStatError);

	buffer = (char *)malloc(statBuffer.st_size);
	ASSERT_RETURN(buffer != NULL, unMemoryError);

	ssize_t got;
	got = read(hFile, buffer, statBuffer.st_size);
	ASSERT_RETURN(got == statBuffer.st_size, unFileError);
	RETURN(unSuccess);

EXIT:
	if (returnValue == unSuccess)
		{
		*bufferOut = buffer;
		}
	else
		{
		if (buffer != NULL)
			free(buffer);
		}
	
	if (hFile != -1)
		close(hFile);

	return returnValue;
	}

#endif





unStatus examineFile(char *filename)
	{
	char *buffer;
	unStatus returnValue;
	ASSERT_SUCCESS(fileGetContents(filename, &buffer));
	if (strstr(buffer, marker) != NULL)
		{
		fprintf(stderr, "error: nocog found \"%s\" marker in %s!\n", marker, filename);
		RETURN(unNocogError);
		}
	RETURN(unSuccess);
EXIT:
	if (buffer != NULL)
		free(buffer);
	return returnValue;
	}


static char *stringSkipWhitespace(char *s)
	{
	while (isspace(*s))
		s++;
	return s;
	}


static char *stringStripWhitespace(char *s)
	{
	char *rv = s;
	char *t = s;
	char *lastNonspace = NULL;
	s = stringSkipWhitespace(s);
	while (*s)
		{
		if (!isspace(*s))
			lastNonspace = s;
		*t++ = *s++;
		}
	if (lastNonspace != NULL)
		*++lastNonspace = 0;
	return rv;
	}



#define IS_EOL(c) ( ((c) == '\n') || ((c) == '\r') )

unStatus handleFileList(char *filename)
	{
	char *buffer;
	char *trace;
	int errorCount = 0;
	unStatus returnValue;

	ASSERT_SUCCESS(fileGetContents(filename, &buffer));

	trace = buffer;
	while (*trace)
		{
		char *filename = trace;
		while (*trace && !IS_EOL(*trace))
			trace++;
		char *eol = trace;
		if (IS_EOL(*trace))
			trace++;
		*eol = 0;
		stringStripWhitespace(filename);
		if (*filename)
			{
			if (examineFile(filename) != unSuccess)
				errorCount++;
			}
		while (*trace && IS_EOL(*trace))
			trace++;
		}

	if (errorCount)
		RETURN(unNocogError);
	RETURN(unSuccess);

EXIT:
	if (buffer != NULL)
		free(buffer);

	return returnValue;
	}






struct ignoredFlag
	{
	char *flag;
	int argumentCount;
	};




ignoredFlag ignoredFlags[] =
	{
		{ "-c", 0 },
		{ "-d", 0 },
		{ "-D", 1 },
		{ "-e", 0 },
		{ "-I", 1 },
		{ "-o", 1 },
		{ "-r", 0 },
		{ "-s", 1 },
		{ "-w", 1 },
		{ "-x", 0 },
		{ "-z", 0 },

		{ NULL, 0 }
	};




int main(int argc, char *argv[])
	{
	int returnValue = 0;
	int failures = 0;
	strcpy(marker, COG_MARKER);
	int wantVersion = (argc >= 2) && (!stricmp(argv[1], "-v"));
	if ((argc < 2) || !stricmp(argv[1], "-h") || wantVersion)
		{
		printf("Nocog 1.0.2 ");
		if (!wantVersion)
			printf("- fast drop-in no-op replacement for Cog\n"
"    Usage: nocog [any & all cog options] [ FILE | @FILELIST ] ...\n"
"\n"
"Nocog scans the listed files for the Cog marker (\"" COG_MARKER "\").\n"
"If it finds it, it prints the name of the offending file and returns -1.\n"
"If it never finds the signature, it returns 0.\n"
"\n"
"You can change the marker with --marker.  (For use with Argent etc.)\n"
"All other command-line options are ignored (well, except -h and -v).\n"
"\n"
"For more information, go to the Nocog homepage:\n"
"    http://www.midwinter.com/~lch/programming/nocog/\n"
);
		printf("\n");
		return 0;
		}

	for (int i = 1; i < argc; i++)
		{
		if ((!stricmp("--marker", argv[i])) && (i < (argc - 1)))
			{
			strcpy(marker, argv[++i]);
			continue;
			}
		for (ignoredFlag *trace = ignoredFlags; trace->flag != NULL; trace++)
			{
			if (!stricmp(trace->flag, argv[i]))
				{
				i += trace->argumentCount;
				goto NEXT_ARGUMENT;
				}
			}

		unStatus status;
		if (argv[i][0] == '@')
			status = handleFileList(argv[i] + 1);
		else
			status = examineFile(argv[i]);
		failures += status != unSuccess;
NEXT_ARGUMENT:
		;
		}

	return (failures) ? -1 : 0;
	}
