/*
** [BEGIN NOTICE]
**
** Copyright (C) 1999-2003 Larry Hastings
**
** This software is provided 'as-is', without any express or implied warranty.
** In no event will the authors be held liable for any damages arising from
** the use of this software.
**
** Permission is granted to anyone to use this software for any purpose,
** including commercial applications, and to alter it and redistribute
** it freely, subject to the following restrictions:
**
** 1. The origin of this software must not be misrepresented; you must not
**    claim that you wrote the original software. If you use this software
**    in a product, an acknowledgment in the product documentation would be
**    appreciated but is not required.
** 2. Altered source versions must be plainly marked as such, and must not be
**    misrepresented as being the original software.
** 3. This notice may not be removed or altered from any source distribution.
**
** The ltimer homepage is here:
**		http://www.midwinter.com/~lch/programming/ltimer/
**
** [END NOTICE]
*/

#include <assert.h>
#include <stdio.h>

#include "ltimer.h"


ltimerUint64 startTime;
ltimer_t timer;
ltimer_t sleeper;
ltimerUint64 totalSleepTime = 0;

// if you're using MSVC 6 and don't have the current Processor Pack installed,
// you can't convert directly from an "unsigned __int64" to a "double".
// Don't ask *me* why; ask the brainy eggheads at Microsoft.  --lch
#define CAST_TO_DOUBLE(uint64) ((double)(__int64)(uint64))

void sleepSomeMore(ltimerUint32 time)
	{
	totalSleepTime += time;
	ltimerSleep(sleeper, time);
	}


void checkAfterSleeping(ltimerUint32 delta)
	{
	sleepSomeMore(delta);
	ltimerUint64 endTime = ltimerGetCurrentTime(timer);
	int deltaTime = (int)((__int64)endTime - (startTime + totalSleepTime));
	assert((deltaTime < 100) && (deltaTime > -100));
	}

void smokeTest(void)
	{
	ltimerCreate(&timer);
	startTime = ltimerGetCurrentTime(timer);

	checkAfterSleeping(1000);
	checkAfterSleeping(2500);
	checkAfterSleeping(1500);

	ltimerDestroy(&timer);
	}





static ltimerUint64 performanceFrequency = 0;
static ltimerUint64 handicapQPC = 0;
static ltimerUint64 handicapRDTSC = 0;

class simpleTimer
	{
	public:
		
		char description[256];
		LARGE_INTEGER startTimeQPC;
		ltimerUint64 startTimeRDTSC;
		bool finished;
		ltimerUint64 elapsedQPC;
		ltimerUint64 elapsedRDTSC;
		
		simpleTimer(char *format, ...)
			{
			va_list list;
			va_start(list, format);
			vsprintf(description, format, list);
			va_end(list);

			finished = false;
			QueryPerformanceCounter(&startTimeQPC);
			startTimeRDTSC = ltimerGetRDTSC();
			}
		
		void finish()
			{
			LARGE_INTEGER endTimeQPC;
			QueryPerformanceCounter(&endTimeQPC);
			ltimerUint64 endTimeRDTSC = ltimerGetRDTSC();

			if (finished)
				return;
			finished = true;

			elapsedQPC = endTimeQPC.QuadPart - (startTimeQPC.QuadPart + handicapQPC);
			elapsedRDTSC = endTimeRDTSC - (startTimeRDTSC + handicapRDTSC);
			printf("%18s: took %f seconds (%I64d machine cycles)\n", description, CAST_TO_DOUBLE(elapsedQPC) / CAST_TO_DOUBLE(performanceFrequency), endTimeRDTSC - startTimeRDTSC);
			}

		~simpleTimer()
			{
			finish();
			}
	};


extern "C" void __fastcall doNothing(void)
	{
	}


void timeNothing(ltimerUint32 iterations)
	{
	simpleTimer timer("nothing x %d", iterations);

	// calling ltimerShutdown() is harmless, and it forces
	// the compiler to generate code for the null loop.
	for (ltimerUint32 i = iterations; i > 0; i--)
		doNothing();
	timer.finish();
	handicapQPC = timer.elapsedQPC;
	handicapRDTSC = timer.elapsedRDTSC;
	}

ltimerUint64 timeLtimer(ltimerUint32 iterations, ltimerUint32 mode)
	{
	ltimer_t ltimer;
	ltimerCreate(&ltimer);
	ltimerSetOption(ltimer, LTIMER_OPTION_TIMING_MODE, mode);

	ltimerUint64 result;

	simpleTimer timer("ltimer x %d", iterations);
	for (ltimerUint32 i = iterations; i > 0; i--)
		result = ltimerGetCurrentTime(ltimer);

	timer.finish();

	ltimerDestroy(&ltimer);

	return timer.elapsedRDTSC;
	}


ltimerUint64 timeTGT(ltimerUint32 iterations)
	{
	simpleTimer timer("TGT x %d", iterations);
	
	ltimerUint32 result;
	for (ltimerUint32 i = iterations; i > 0; i--)
		result = timeGetTime();
	timer.finish();
	
	return timer.elapsedRDTSC;
	}


ltimerUint64 timeQPC(ltimerUint32 iterations)
	{
	simpleTimer timer("QPC x %d", iterations);
	
	LARGE_INTEGER result;
	for (ltimerUint32 i = iterations; i > 0; i--)
		QueryPerformanceCounter(&result);
	timer.finish();

	return timer.elapsedRDTSC;
	}


ltimerUint64 timeRDTSC(ltimerUint32 iterations)
	{
	simpleTimer timer("RDTSC x %d", iterations);
	
	ltimerUint64 result;
	for (ltimerUint32 i = iterations; i > 0; i--)
		result = ltimerGetRDTSC();
	timer.finish();

	return timer.elapsedRDTSC;
	}


void performanceTest(char *argv1)
	{
	LARGE_INTEGER largeInteger;
	QueryPerformanceFrequency(&largeInteger);
	performanceFrequency = largeInteger.QuadPart;
	printf("QPC's resolution is %I64d cycles per second.\n", performanceFrequency);

	ltimerUint32 iterations = 1000 * 1000;
	if (argv1 != NULL)
		iterations = strtoul(argv1, NULL, 10);

	timeNothing(iterations);
	printf("From now on, we'll subtract that handicap.\n");
	printf("\n");
	
	printf("Forcing ltimer to use its slow method...\n");
	ltimerUint64 ltimer1 = timeLtimer(iterations, LTIMER_TIMING_MODE_SAFETY);
	ltimerUint64 tgt = timeTGT(iterations);
	ltimerUint64 qpc = timeQPC(iterations);
	ltimerUint64 rdtsc = timeRDTSC(iterations);

	printf("\n");
	printf("Sleeping to give ltimer time to calibrate, then using its fast method...\n");
	ltimerUint32 sleepMs;
	ltimerGetDefaultOption(LTIMER_OPTION_CALIBRATION_PERIOD, &sleepMs);
	ltimerSleep(sleeper, sleepMs);
	ltimerUint64 ltimer2 = timeLtimer(iterations, LTIMER_TIMING_MODE_PERFORMANCE);
	
	printf("\n");
	printf("When it hasn't warmed up yet, ltimer is %1.3fx slower than QPC.\n", CAST_TO_DOUBLE(ltimer1) / CAST_TO_DOUBLE(qpc));
	printf("But once it's warmed up, ltimer is %1.3fx faster than QPC!\n", CAST_TO_DOUBLE(qpc) / CAST_TO_DOUBLE(ltimer2));
	}


int main(int argc, char *argv[])
	{
	ltimerStartup();
	ltimerCreate(&sleeper);
	smokeTest();

	char *argv1 = (argc > 1) ? argv[1] : NULL;
	performanceTest(argv1);

/*	
	Sleep(35 * 60 * 1000);
	printf("ltimer says the time is %I64u, and RDTSC gets %I64u cycles/sec.\n", ltimerGetCurrentTime(sleeper), ltimerGetRdtscFrequency(sleeper));
	Sleep(1 * 60 * 1000);
	printf("ltimer says the time is %I64u, and RDTSC gets %I64u cycles/sec.\n", ltimerGetCurrentTime(sleeper), ltimerGetRdtscFrequency(sleeper));
*/

	ltimerDestroy(&sleeper);
	
	ltimerShutdown();

	return 0;
	}


