/*
** [BEGIN NOTICE]
**
** Copyright (C) 1999-2003 Larry Hastings
**
** This software is provided 'as-is', without any express or implied warranty.
** In no event will the authors be held liable for any damages arising from
** the use of this software.
**
** Permission is granted to anyone to use this software for any purpose,
** including commercial applications, and to alter it and redistribute
** it freely, subject to the following restrictions:
**
** 1. The origin of this software must not be misrepresented; you must not
**    claim that you wrote the original software. If you use this software
**    in a product, an acknowledgment in the product documentation would be
**    appreciated but is not required.
** 2. Altered source versions must be plainly marked as such, and must not be
**    misrepresented as being the original software.
** 3. This notice may not be removed or altered from any source distribution.
**
** The ltimer homepage is here:
**		http://www.midwinter.com/~lch/programming/ltimer/
**
** [END NOTICE]
*/

/*
** ltimer 1.0.1
**
** A machine-speed-independent timer class.
**
** ltimer uses a two-tiered approach to determine the current time.
** For the first LTIMER_SECONDS_UNTIL_USE_RDTSC seconds, it determines
** the time using QueryPerformanceCounter().  After that, it calculates
** what the local machine's clock frequency must be, and uses the
** Pentium-and-above RDTSC instruction.  It then recalculates the
** clock frequency after every LTIMER_RESYNC_RDTSC_FREQUENCY_EVERY calls,
** so it gets even more accurate over time.
**
** For information on using RDTSC, peep this my homies:
**   http://cedar.intel.com/software/idap/media/pdf/rdtscpm1.pdf
** or search the Intel developer site for "rdtsc cpuid timer"
** and wade through the results.
**
** ltimer also prevents possible "retrograde" time observation.
** Every now and then I observe QueryPerformanceCounter() actually
** return a *smaller* value than the time before.  *shudder*
** This causes all sorts of conniptions in my timing loops.
** Anyway, ltimer prevents you from seeing this.  If QPC() ever
** reports a value smaller than the one it returned last time,
** ltimer ignores QPC() and uses the previous returned value.
** Your observed change in time will be 0, which I feel is an
** improvement over a negative number.  It doesn't bother with
** this for values returned by RDTSC, as I've never seen retrograde
** values with RDTSC.  (It seems far less likely, as RDTSC is
** returning the CPU's raw cycle count.)
**
** (Warning: if your software actually travels back in time,
** either disable this feature or do not use ltimer.)
**
*/


#ifndef __LTIMER_H
#define __LTIMER_H

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */


#include <windows.h>

struct ltimer_s;
typedef struct ltimer_s ltimer_s;
typedef ltimer_s *ltimer_t;

#define LTIMER_FUNCTION  __fastcall

typedef unsigned __int64 ltimerUint64;
typedef unsigned     int ltimerUint32;

extern      HRESULT LTIMER_FUNCTION ltimerStartup(void);
extern      HRESULT LTIMER_FUNCTION ltimerShutdown(void);

extern      HRESULT LTIMER_FUNCTION ltimerGetDefaultOption(ltimerUint32 option, ltimerUint32 *value);
extern      HRESULT LTIMER_FUNCTION ltimerSetDefaultOption(ltimerUint32 option, ltimerUint32 value);


extern      HRESULT LTIMER_FUNCTION ltimerCreate(ltimer_t *ltimer);
extern      HRESULT LTIMER_FUNCTION ltimerDestroy(ltimer_t *ltimer);
extern      HRESULT LTIMER_FUNCTION ltimerReset(ltimer_t ltimer);

extern ltimerUint64 LTIMER_FUNCTION ltimerGetCurrentTime(ltimer_t ltimer);
extern ltimerUint64 LTIMER_FUNCTION ltimerGetLastReportedTime(ltimer_t ltimer);
extern ltimerUint64 LTIMER_FUNCTION ltimerGetRdtscFrequency(ltimer_t ltimer);

extern      HRESULT LTIMER_FUNCTION ltimerSleep(ltimer_t ltimer, ltimerUint64 ticks);
extern      HRESULT LTIMER_FUNCTION ltimerSleepUntil(ltimer_t ltimer, ltimerUint64 time);


extern      HRESULT LTIMER_FUNCTION ltimerGetOption(ltimer_t ltimer, ltimerUint32 option, ltimerUint32 *value);
extern      HRESULT LTIMER_FUNCTION ltimerSetOption(ltimer_t ltimer, ltimerUint32 option, ltimerUint32 value);

	#define LTIMER_OPTION_INVALID               (0)
	#define LTIMER_OPTION_CALIBRATION_PERIOD    (1) /* in ms */
	#define LTIMER_OPTION_RECALIBRATION_PERIOD  (2) /* in ms */
	#define LTIMER_OPTION_LAST_GLOBAL_ONLY      (2)

	#define LTIMER_OPTION_TIMER_RESOLUTION      (3)
	#define LTIMER_OPTION_TIMING_MODE           (4) /* see below */
	#define LTIMER_OPTION_LONG_SLEEP_THRESHOLD  (5) /* in ms */
	#define LTIMER_OPTION_SHORT_SLEEP_THRESHOLD (6) /* in ms */

	#define LTIMER_OPTION_LAST                  (6)


	#define LTIMER_TIMING_MODE_INVALID      (0)
	#define LTIMER_TIMING_MODE_PERFORMANCE  (1)
	#define LTIMER_TIMING_MODE_SAFETY       (2)





#ifdef __cplusplus


inline __declspec(naked) ltimerUint64 ltimerGetRDTSC(void)
	{
	__asm
		{
		push ebx    // cpuid overwrites ebx
		push ecx    // and ecx

		// CPUID: flush the out-of-order instruction pipeline
		__asm __emit 0x0F __asm __emit 0xA2

		// RDTSC: and get the current cycle into edx:eax
		__asm __emit 0x0F __asm __emit 0x31

		pop ecx
		pop ebx

		ret
		}
	}

};
#endif /* __cplusplus */

#endif /* __LTIMER_H */
