/*
** [BEGIN NOTICE]
**
** JSON Library Copyright (C) 2006-2007 Larry Hastings
**
** This software is provided 'as-is', without any express or implied warranty.
** In no event will the authors be held liable for any damages arising from
** the use of this software.
**
** Permission is granted to anyone to use this software for any purpose,
** including commercial applications, and to alter it and redistribute
** it freely, subject to the following restrictions:
**
** 1. The origin of this software must not be misrepresented; you must not
**    claim that you wrote the original software. If you use this software
**    in a product, an acknowledgment in the product documentation would be
**    appreciated but is not required.
** 2. Altered source versions must be plainly marked as such, and must not be
**    misrepresented as being the original software.
** 3. This notice may not be removed or altered from any source distribution.
**
** My JSON library's homepage is here:
**              http://www.midwinter.com/~lch/programming/json/
**
** [END NOTICE]
*/

#include <assert.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>


//
// To support the extended syntax for binary objects in JavaScript, add this to your code:
/*
function jsonClass()
	{
	this.bx = bx;

	function bx(s)
		{
		var hexDigits = "0123456789abcdef";
		var returnString = "";
		s = s.toLowerCase();
		for (var i = 0; i < s.length; i += 2)
			{
			var substr = s.substring(i, i+2);
			if (substr.length != 2)
				break;
			var firstDigit = hexDigits.indexOf(substr[0]);
			var secondDigit = hexDigits.indexOf(substr[1]);
			if ((firstDigit == -1) || (secondDigit == -1))
				break;
			var character = String.fromCharCode((firstDigit * 16) +  secondDigit);
			returnString = returnString + character;
			}
		return returnString;
		}
	}
json = new jsonClass();
*/
//


//
// Simple JSON scanner & parser & formatter and (someday) data type.
//
// http://www.json.org/
//
// Scanner:
// I *slightly* extended the JSON syntax:
//	* # and // are line comments
//	* /* .. */ is a block comment
//	* binary data:
//		"x" : json.bx("414243"), 
//    the name bx is meant to connote "binary in hex".  it's also
//    supposed to be short.  (it is far more successful at the latter
//    than the former.)  the string is interpreted as hexadecimal;
//	  the above example would turn into the string "ABC".
//
//
// Parser:
// explicitly allows trailing comma at the end of an "object".
// example:
//        { "a" : "x", "b" : "y", "c" : "z", }
//                                         ^ allowed!
//
// Formatter:
//	* can output to one of three formats:
//		* no whitespace
//		* single spaces only
//		* full indent with newlines
//			* in this form, you can control the indent and newline strings
//	* configurable line- and block-comment syntax
//	* if you don't want it to support block comments,
//		set blockCommentStartString to NULL


//
// cheap dynamic string class
//
struct dynamicString
	{
	char *buffer;
	int length; // total bytes currently stored
	int size; // total bytes allocated

	dynamicString()
		{
		size = 256;
		buffer = (char *)malloc(size);
		reset();
		}

	~dynamicString()
		{
		free(buffer);
		}

private:
	void ensureSize(int wantedSize)
		{
		while (wantedSize >= size)
			size = (size * 3) / 2;
		buffer = (char *)realloc(buffer, size);
		}

	void ensureOffset(char *&trace)
		{
		int offset = trace - buffer;
		ensureSize(offset);
		trace = buffer + offset;
		}
public:

	void reset(void)
		{
		*buffer = 0;
		length = 0;
		}

	void memcpy(const char *s, int length)
		{
		ensureSize(length);
		::memcpy(buffer, s, length);
		buffer[length] = 0;
		this->length = length;
		}

	void strcpy(const char *s)
		{
		this->memcpy(s, strlen(s));
		}

	void memcat(const char *s, int length)
		{
		int currentLength = this->length;
		ensureSize(length + currentLength + 1);
		::memcpy(buffer + currentLength, s, length);
		buffer[currentLength + length] = 0;
		this->length += length;
		}

	void strcat(const char *s)
		{
		memcat(s, strlen(s));
		}

	void charcat(const char c)
		{
		memcat(&c, 1);
		}

	operator char *(void)
		{
		return buffer;
		}
	};

//
// the JSON token
//

struct jsonHardcodedString
	{
	char *string;
	int length;
	inline bool matches(const char *s)
		{
		return !strnicmp(s, string, length);
		}
	inline void skip(const char *&s)
		{
		s += length;
		}
	};


//
// the below definitions (jsonToken etc)
// were generated by the following program.
// DON'T CHANGE THEM BY HAND.
// instead, copy the program below into a new file,
// make your changes there, compile and run, and
// copy the output of the program in below there.
// (and copy your source back into the comment here, too.)
//
//
// the program:
//
/*

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define TOKENS \
	TOKEN(Invalid) \
	TOKEN(EOF) \
	TOKEN(String) \
	TOKEN(Number) \
	TOKEN(ObjectEnter) \
	TOKEN(ObjectLeave) \
	TOKEN(ArrayEnter) \
	TOKEN(ArrayLeave) \
	TOKEN(Comma) \
	TOKEN(Colon) \
	TOKEN(True) \
	TOKEN(False) \
	TOKEN(Null) \
	TOKEN(LineComment) \
	TOKEN(Binary) \
	TOKEN(Json) \
	TOKEN(Dot) \
	TOKEN(OpenParentheses) \
	TOKEN(CloseParentheses) \
	TOKEN(BinaryInHex) \

enum jsonToken
	{
	#define TOKEN(name) json ## name,
	TOKENS
	#undef TOKEN
	};


static char *tokens[256];

static jsonToken tokenLookupTable[256];
static jsonToken numberLookupTable[256];
static char *tokenStrings[256];

int main(int argc, char *argv[])
	{
	int i;
	memset(tokens, 0, sizeof(tokens));

	printf("/////// 8X -- cut here -- X8 ///////\n");

	int tokenCount = 0;
#define TOKEN(name) tokens[tokenCount++] = #name;
	TOKENS
	#undef TOKEN

	printf(
"enum jsonToken\n"
"\t{\n"
		);

	for (i = 0; i < tokenCount; i++)
		printf("\tjson%s = %i,\n", tokens[i], i);

	printf(
"\n"
"\tjsonFourByteTrick = 0x10000000\n"
"\t};\n"
"\n"
		);

	printf(
"static char *tokenDescriptions[%d] =\n"
"\t{\n"
		,
		tokenCount
		);

	for (i = 0; i < tokenCount; i++)
		printf("\t\"%s\",\n", tokens[i]);

	printf(
"\t};\n"
"\n"
"\n"
"const char *jsonDescribeToken(jsonToken t)\n"
"\t{\n"
"\tint token = (int)t;\n"
"\tif (token >= %d)\n"
"\t\treturn \"*unrecognized!*\";\n"
"\treturn tokenDescriptions[token];\n"
"\t}\n"
"\n"
		, tokenCount
		);


	memset(tokenLookupTable, jsonInvalid, sizeof(tokenLookupTable));
	memset(tokenStrings, 0, sizeof(tokenStrings));

	for (i = '0'; i <= '9'; i++)
		{
		tokenLookupTable[i] = jsonNumber;
		}
	tokenLookupTable['-'] = jsonNumber;
	tokenLookupTable['+'] = jsonNumber;

	memcpy(numberLookupTable, tokenLookupTable, sizeof(numberLookupTable));
	numberLookupTable['.'] = jsonNumber;
	numberLookupTable['E'] = jsonNumber;

	tokenLookupTable['"'] = jsonString;
	
	tokenLookupTable['{'] = jsonObjectEnter;
	tokenLookupTable['}'] = jsonObjectLeave;
	tokenLookupTable['['] = jsonArrayEnter;
	tokenLookupTable[']'] = jsonArrayLeave;
	tokenLookupTable[','] = jsonComma;
	tokenLookupTable[':'] = jsonColon;
	
	tokenLookupTable['t'] = jsonTrue;
	tokenLookupTable['T'] = jsonTrue;
	tokenStrings[jsonTrue] = "true";
	
	tokenLookupTable['f'] = jsonFalse;
	tokenLookupTable['F'] = jsonFalse;
	tokenStrings[jsonFalse] = "false";
	
	tokenLookupTable['n'] = jsonNull;
	tokenLookupTable['N'] = jsonNull;
	tokenStrings[jsonNull] = "null";

	tokenLookupTable[0] = jsonEOF;

	// extensions
	tokenLookupTable['#'] = jsonLineComment;
	tokenLookupTable['/'] = jsonLineComment;

	tokenLookupTable['j'] = jsonJson;
	tokenStrings[jsonJson] = "json";
	tokenLookupTable['.'] = jsonDot;
	tokenLookupTable['b'] = jsonBinaryInHex;
	tokenStrings[jsonBinaryInHex] = "bx";
	tokenLookupTable['('] = jsonOpenParentheses;
	tokenLookupTable[')'] = jsonCloseParentheses;

	printf(
"static jsonToken tokenLookupTable[256] =\n"
"\t{\n"
"\t"
		);

	for (i = 0; i < 256; i++)
		{
		char buffer[64];
		sprintf(buffer, "json%s,", tokens[tokenLookupTable[i]]);
		printf("%23s", buffer);
		if ((i & 3) == 3)
			printf("\n\t");
		}

	printf(
"};\n"
"\n"
		);

	printf(
"static jsonToken numberLookupTable[256] =\n"
"\t{\n"
"\t"
		);

	for (i = 0; i < 256; i++)
		{
		char buffer[64];
		sprintf(buffer, "json%s,", tokens[numberLookupTable[i]]);
		printf("%12s", buffer);
		if ((i % 5) == 4)
			printf("\n\t");
		}

	printf(
"\n"
"\t};\n"
"\n"
		);

	printf(
"static jsonHardcodedString tokenStrings[%d] =\n"
"\t{\n"
		,
		tokenCount
		);

	for (i = 0; i < tokenCount; i++)
		{
		char *s = tokenStrings[i];
		if (s == NULL)
			printf("\t{ NULL, 0 },\n");
		else
			printf("\t{ \"%s\", %d },\n", s, strlen(s));
		}

	printf(
"\t};\n"
"\n"
		);

	printf("/////// 8X -- cut here -- X8 ///////\n");

	return 0;
	}

*/


//
// the output of the above program:
//

/////// 8X -- cut here -- X8 ///////
enum jsonToken
	{
	jsonInvalid = 0,
	jsonEOF = 1,
	jsonString = 2,
	jsonNumber = 3,
	jsonObjectEnter = 4,
	jsonObjectLeave = 5,
	jsonArrayEnter = 6,
	jsonArrayLeave = 7,
	jsonComma = 8,
	jsonColon = 9,
	jsonTrue = 10,
	jsonFalse = 11,
	jsonNull = 12,
	jsonLineComment = 13,
	jsonBinary = 14,
	jsonJson = 15,
	jsonDot = 16,
	jsonOpenParentheses = 17,
	jsonCloseParentheses = 18,
	jsonBinaryInHex = 19,

	jsonFourByteTrick = 0x10000000
	};

static char *tokenDescriptions[20] =
	{
	"Invalid",
	"EOF",
	"String",
	"Number",
	"ObjectEnter",
	"ObjectLeave",
	"ArrayEnter",
	"ArrayLeave",
	"Comma",
	"Colon",
	"True",
	"False",
	"Null",
	"LineComment",
	"Binary",
	"Json",
	"Dot",
	"OpenParentheses",
	"CloseParentheses",
	"BinaryInHex",
	};


const char *jsonDescribeToken(jsonToken t)
	{
	int token = (int)t;
	if (token >= 20)
		return "*unrecognized!*";
	return tokenDescriptions[token];
	}

static jsonToken tokenLookupTable[256] =
	{
	               jsonEOF,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,            jsonString,       jsonLineComment,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	   jsonOpenParentheses,  jsonCloseParentheses,           jsonInvalid,            jsonNumber,
	             jsonComma,            jsonNumber,               jsonDot,       jsonLineComment,
	            jsonNumber,            jsonNumber,            jsonNumber,            jsonNumber,
	            jsonNumber,            jsonNumber,            jsonNumber,            jsonNumber,
	            jsonNumber,            jsonNumber,             jsonColon,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,             jsonFalse,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,              jsonNull,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	              jsonTrue,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,        jsonArrayEnter,
	           jsonInvalid,        jsonArrayLeave,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,       jsonBinaryInHex,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,             jsonFalse,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,              jsonJson,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,              jsonNull,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	              jsonTrue,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,       jsonObjectEnter,
	           jsonInvalid,       jsonObjectLeave,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	           jsonInvalid,           jsonInvalid,           jsonInvalid,           jsonInvalid,
	};

static jsonToken numberLookupTable[256] =
	{
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid, jsonNumber,jsonInvalid,
	 jsonNumber, jsonNumber,jsonInvalid, jsonNumber, jsonNumber,
	 jsonNumber, jsonNumber, jsonNumber, jsonNumber, jsonNumber,
	 jsonNumber, jsonNumber, jsonNumber,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid, jsonNumber,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,jsonInvalid,
	jsonInvalid,
	};

static jsonHardcodedString tokenStrings[20] =
	{
	{ NULL, 0 },
	{ NULL, 0 },
	{ NULL, 0 },
	{ NULL, 0 },
	{ NULL, 0 },
	{ NULL, 0 },
	{ NULL, 0 },
	{ NULL, 0 },
	{ NULL, 0 },
	{ NULL, 0 },
	{ "true", 4 },
	{ "false", 5 },
	{ "null", 4 },
	{ NULL, 0 },
	{ NULL, 0 },
	{ "json", 4 },
	{ NULL, 0 },
	{ NULL, 0 },
	{ NULL, 0 },
	{ "bx", 2 },
	};

/////// 8X -- cut here -- X8 ///////


//
// the JSON scanner
// 



#define BACKSLASH_CHARACTERS \
	BACKSLASH('b', '\b') \
	BACKSLASH('f', '\f') \
	BACKSLASH('n', '\n') \
	BACKSLASH('r', '\r') \
	BACKSLASH('t', '\t') \



struct jsonScanner
	{
	jsonToken token;
	jsonToken pushedToken;

	dynamicString buffer;
	int length;

	const char *description;

	const char *trace;

private:
	void initialize(void)
		{
		token = pushedToken = jsonInvalid;
		description = "";
		}
public:

	jsonScanner(void)
		{
		setInput("");
		initialize();
		}

	jsonScanner(const char *input)
		{
		setInput(input);
		initialize();
		}

	void setInput(const char *input)
		{
		this->trace = input;
		}

	void eof(void)
		{
		setInput("");
		token = jsonEOF;
		}

	void pushToken(jsonToken token)
		{
		pushedToken = token;
		}

	jsonToken next(void)
		{
		if (pushedToken != jsonInvalid)
			{
			token = pushedToken;
			pushedToken = jsonInvalid;
			return token;
			}

START_OVER:

		for (;;)
			{
			token = tokenLookupTable[*trace];
			if (token != jsonInvalid)
				break;
			trace++;
			}
		buffer.reset();

		switch (token)
			{
			default:
				trace++;
				break;

			case jsonEOF:
				break;

			case jsonLineComment:
				if (*trace == '/')
					{
					trace++;
					switch (*trace)
						{
						case '/':
							break;

						default:
							eof();
							break;

						case '*':
							trace++;
							// C-style block comment!
							while (*trace && strncmp(trace, "*/", 2 * sizeof(char)))
								trace++;
							// if we found the end-comment string, skip it.
							if (*trace)
								trace += 2;
							goto START_OVER;
						}
					}
				while ((*trace != '\r') && (*trace != '\n'))
					trace++;
				goto START_OVER;

			case jsonString:
				trace++;
				for (;;)
					{
					switch (*trace)
						{
						case 0:
						case '"':
							goto END_STRING;

						default:
							buffer.charcat(*trace++);
							break;

						case '\\':
							trace++;
							switch (*trace)
								{
								default:
									buffer.charcat(*trace++);
									break;

								#define BACKSLASH(character, mappedCharacter ) \
								case character: \
									buffer.charcat(mappedCharacter); \
									trace++; \
									break; \

								BACKSLASH_CHARACTERS
								#undef BACKSLASH
								case 'u':
									{
									trace++;
									int length = strlen(trace);
									if (length < 4)
										{
										eof();
										continue;
										}

									char minibuffer[5];
									memcpy(minibuffer, trace, 4);
									minibuffer[4] = 0;
									buffer.charcat((char)strtoul(minibuffer, NULL, 16));
									trace += 4;
									break;
									}
								}
						}
					}
	END_STRING:
				if (*trace)
					trace++;
				break;

			case jsonNumber:
				while (numberLookupTable[*trace] == jsonNumber)
					buffer.charcat(*trace++);
				break;

			case jsonJson:
				{
				jsonHardcodedString *s = tokenStrings + (int)jsonJson;
				if (!s->matches(trace))
					eof();
				else
					{
					s->skip(trace);

					#define EAT(t) \
						next(); \
						if (token != t) \
							{ \
							eof(); \
							break; \
							} \

					// parse everything first, so we know it is correctly-formed.
					EAT(jsonDot);
					EAT(jsonBinaryInHex);
					s = tokenStrings + (int)jsonBinaryInHex;
					if (!s->matches(trace))
						{
						eof();
						break;
						}
					EAT(jsonOpenParentheses);
					EAT(jsonString);
					dynamicString hexString;
					hexString.strcpy(buffer);
					EAT(jsonCloseParentheses);

					#undef EAT

					// copy over the binary data
					static char *hexDigits = "0123456789abcdef";
					buffer.reset();
					for (int i = 0; i < hexString.length; i += 2)
						{
						char *firstDigit = strchr(hexDigits, tolower(hexString[i]));
						char *secondDigit = strchr(hexDigits, tolower(hexString[i + 1]));
						if ((firstDigit == NULL) || (secondDigit == NULL))
							break;
						char character = ((firstDigit - hexDigits) * 16) + (secondDigit - hexDigits);
						buffer.charcat(character);
						}

					// finally, since we recursed to parse the sub-tokens of the binary field,
					// we've just stomped all over token and length.  so fix 'em.
					token = jsonBinary;
					length = buffer.length;
					description = jsonDescribeToken(jsonBinary);
					return token;
					}
				break;
				}

			case jsonTrue:
			case jsonFalse:
			case jsonNull:
				{
				jsonHardcodedString *s = tokenStrings + (int)token;
				if (!s->matches(trace))
					eof();
				else
					s->skip(trace);
				break;
				}
			}

		length = buffer.length;

		description = jsonDescribeToken(token);
		return token;
		}
	};



//
// JSON parser
//

#define PARSER_TYPES \
	PARSER_TYPE(Invalid, 0) \
	PARSER_TYPE(EOF, 1) \
	\
	PARSER_TYPE(String, 100) \
	PARSER_TYPE(Integer, 101) \
	PARSER_TYPE(Integer64, 102) \
	PARSER_TYPE(FloatingPoint, 103) \
	PARSER_TYPE(Boolean, 104) \
	PARSER_TYPE(Null, 105) \
	\
	PARSER_TYPE(Binary, 200) \
	\
	PARSER_TYPE(ArrayEnter, 300) \
	PARSER_TYPE(ArrayLeave, 301) \
	PARSER_TYPE(ObjectEnter, 302) \
	PARSER_TYPE(ObjectLeave, 303) \


enum jsonParserType
	{
#define PARSER_TYPE(name, value) jsonpt ## name = value,
	PARSER_TYPES
#undef PARSER_TYPE

	jsonptFourBytes = 0x10000000
	};

const char *jsonDescribeParserType(jsonParserType type)
	{
	switch (type)
		{
#define PARSER_TYPE(name, value) case jsonpt ## name: return #name ; // " (" #value ")";
		PARSER_TYPES
#undef PARSER_TYPE

		default:
			return "*unrecognized parser type!*";
		}
	}



struct jsonParserState
	{
	jsonParserType type;
	int level;
	struct jsonParserState *next;
	};

struct jsonParser
	{
	dynamicString name;

	jsonParserType type;
	jsonParserState *state;
	jsonParserState *nextState;

	dynamicString stringOrBinary;
	int stringOrBinaryLength;

	int integer;
	__int64 integer64;
	double floatingPoint;
	bool boolean;

	jsonScanner scanner;

	jsonParserState _bottomState;


private:
	void initialize(void)
		{
		type = jsonptInvalid;

		state = &_bottomState;
		_bottomState.level = 0;
		_bottomState.next = NULL;
		_bottomState.type = jsonptInvalid;
		nextState = NULL;
		}
public:

	jsonParser(const char *input)
		{
		scanner.setInput(input);
		initialize();
		}

	jsonParser(void)
		{
		initialize();
		}


	void eof(void)
		{
		scanner.eof();
		type = jsonptEOF;
		}

	void popState(void)
		{
		if (state == &_bottomState)
			{
			eof();
			return;
			}
		jsonParserState *next = state->next;
		delete state;
		state = next;
		}

	void freeStates(void)
		{
		while (state != &_bottomState)
			popState();
		}

	~jsonParser(void)
		{
		freeStates();
		}

	jsonParserType next()
		{
		#define EAT(t) \
			if (scanner.next() != t) \
				{ \
				eof(); \
				return type; \
				} \

		// clear old values:
		*name = 0;
		*stringOrBinary = 0;
		stringOrBinaryLength = 0;
		integer = 0;
		integer64 = 0;
		floatingPoint = 0.0;


		// what cleanup do we have from last time?
		switch (type)
			{
			case jsonptObjectEnter:
			case jsonptArrayEnter:
				{
				if (nextState == NULL)
					{
					eof();
					return type;
					}
				state = nextState;
				nextState = NULL;
				break;
				}

			case jsonptObjectLeave:
			case jsonptArrayLeave:
				{
				popState();
				break;
				}
			}
		// if we're inside an object or array.
		if (state != &_bottomState)
			{
			// provisionally eat the between-element comma here
			if (scanner.next() != jsonComma)
				scanner.pushToken(scanner.token);

			switch (state->type)
				{
				case jsonptObjectEnter:
					// get the "name" of the current value
					scanner.pushToken(scanner.next());
					if (scanner.token != jsonObjectLeave)
						{
						EAT(jsonString);
						name.strcpy(scanner.buffer);
						EAT(jsonColon);
						}
					break;

				case jsonptArrayEnter:
					break;
				}
			}

		//
		// reading in a value.
		//

		#define MUST_BE_NESTED \
			if (state == &_bottomState) \
				{ \
				eof(); \
				break; \
				} \

NEXT_TOKEN:
		switch (scanner.next())
			{
			case jsonComma:
				goto NEXT_TOKEN;

			case jsonString:
				MUST_BE_NESTED;
				stringOrBinary.strcpy(scanner.buffer);
				type = jsonptString;
				break;

			case jsonBinary:
				MUST_BE_NESTED;
				stringOrBinary.memcpy(scanner.buffer, scanner.length);
				stringOrBinaryLength = scanner.length;
				type = jsonptBinary;
				break;

			case jsonNumber:
				{
				MUST_BE_NESTED;
				// Aha!  But what *kind* of number?
				if ((strchr(scanner.buffer, '.') != NULL)
					|| (strchr(scanner.buffer, 'E') != NULL)
					|| (strchr(scanner.buffer, 'e') != NULL))
					{
					// it contains ".", "e", or "E", therefore it is
					// a floating-point number.  Theoretically it could
					// be 1E5, or 10000, which could be expressed as an
					// integer.  But I don't think that's the intent of
					// 1E5, and in any case the spirit is unwilling.
					floatingPoint = strtod(scanner.buffer, NULL);
					type = jsonptFloatingPoint;
					break;
					}

				// okay, must be an integer.
				integer64 = _atoi64(scanner.buffer);

				if ((integer64 <= INT_MAX) && (integer64 >= INT_MIN))
					{
					integer = (int)integer64;
					type = jsonptInteger;
					}
				else
					type = jsonptInteger64;
				break;
				}

			case jsonNull:
				type = jsonptNull;
				break;

			case jsonTrue:
			case jsonFalse:
				type = jsonptBoolean;
				boolean = (type == jsonTrue);
				break;

			case jsonObjectEnter:
			case jsonArrayEnter:
				{
				nextState = new jsonParserState;
				nextState->next = state;
				nextState->type = type = (scanner.token == jsonObjectEnter) ? jsonptObjectEnter : jsonptArrayEnter;
				nextState->level = state->level + 1;
				type = (scanner.token == jsonObjectEnter) ? jsonptObjectEnter : jsonptArrayEnter;
				break;
				}

			case jsonObjectLeave:
			case jsonArrayLeave:
				{
				MUST_BE_NESTED;
				if (
					((scanner.token == jsonObjectLeave) && (state->type != jsonptObjectEnter))
					|| ((scanner.token == jsonArrayLeave) && (state->type != jsonptArrayEnter))
					)
					{
					eof();
					break;
					}
				type = (scanner.token == jsonObjectLeave) ? jsonptObjectLeave : jsonptArrayLeave;
				break;
				}

			default:
				eof();
				break;
			}

		return type;
		}
	};



//
//
// The JSON formatter
//
//

enum jsonFormatterStyle
	{
	jsonfsInvalid = 0,
	jsonfsNoWhitespace = 1,
	jsonfsOneSpace = 2,
	jsonfsTabsAndNewlines = 3,

	jsonfsFourByteTrick = 0x10000000
	};

struct jsonFormatter
	{
	dynamicString output;
	jsonFormatterStyle style;
	jsonParserState *state;
	char *indentString;
	char *eolString;
	char *lineCommentString;
	char *blockCommentStartString;
	char *blockCommentLineStartString;
	char *blockCommentEndString;
	bool inComment;

	bool wantComma;

	jsonFormatter(void)
		{
		initialize(jsonfsTabsAndNewlines);
		}

	jsonFormatter(jsonFormatterStyle style)
		{
		initialize(style);
		}

	~jsonFormatter(void)
		{
		while (state != NULL)
			popState();
		}


private:
	void initialize(jsonFormatterStyle style)
		{
		this->style = style;
		state = NULL;
		genericEnter(jsonptObjectEnter);
		output.strcpy("{");
		wantComma = false;

		indentString = "\t";
		eolString = "\n";
		lineCommentString = "//";
		blockCommentStartString = "/*" ;
		blockCommentLineStartString = "** ";
		blockCommentEndString = "*/";
		inComment = false;
		}

	void popState(void)
		{
		if (state != NULL)
			{
			jsonParserState *old = state;
			state = state->next;
			delete old;
			}
		}

	void genericEnter(jsonParserType type)
		{
		jsonParserState *s = new jsonParserState;
		s->next = state;
		s->type = type;
		s->level = (state != NULL) ? (state->level + 1) : 0;
		state = s;
		}

	bool genericLeave(jsonParserType token)
		{
		if (token != state->type)
			return false;
		popState();
		return true;
		}

	void startLine(void)
		{
		switch (style)
			{
			default:
			case jsonfsNoWhitespace:
				break;
			case jsonfsOneSpace:
				output.strcat(" ");
				break;
			case jsonfsTabsAndNewlines:
				output.strcat(eolString);
				for (int i = 0; i < state->level; i++)
					output.strcat(indentString);
				break;
			}
		}

	void internalSpacing(void)
		{
		switch (style)
			{
			default:
			case jsonfsNoWhitespace:
				output.strcat("");
				break;
			case jsonfsTabsAndNewlines:
			case jsonfsOneSpace:
				output.strcat(" ");
				break;
			}
		}

	static void appendQuotedString(dynamicString *string, const char *s)
		{
		string->charcat('\"');
		while (*s)
			{
			// if in the normal range of printable ASCII characters, and not a character used in quoting
			if ((*s >= ' ') && (*s <= '~') && (*s != '"') && (*s != '\\'))
				// spit it out
				string->charcat(*s++);
			else
				{
				// handle all special cases:
				switch (*s)
					{
					case '"':
						string->strcat("\\\"");
						break;
					case '\\':
						string->strcat("\\\\");
						break;

					#define BACKSLASH(character, mappedCharacter ) \
					case mappedCharacter: \
						{ \
						static const char quoted[3] = { '\\', character, 0 }; \
						string->strcat(quoted); \
						break; \
						} \

					BACKSLASH_CHARACTERS
					#undef BACKSLASH

					default:
						{
						char hexCharacters[9];
						sprintf(hexCharacters, "%08x", (unsigned int)(unsigned char)*s);
						string->strcat("\\u");
						string->strcat(hexCharacters + 4);
						}
					}
				s++;
				}
			}
		string->charcat('\"');
		}

	bool internalPrint(const char *name, const char *value, int valueLength = -1, bool quoted=false, bool stateMustNotBeNull=true)
		{
		if (stateMustNotBeNull && (state != NULL))
			{
			if (name == NULL)
				{
				if (state->type == jsonptObjectEnter)
					return false;
				}
			else
				{
				if (state->type == jsonptArrayEnter)
					return false;
				}
			}

		if (valueLength == -1)
			valueLength = strlen(value);

		if (wantComma)
			output.charcat(',');
		wantComma = true;

		if (inComment)
			{
			inComment = false;
			startLine();
			output.strcat(blockCommentEndString);
			}

		startLine();
		if (name != NULL)
			{
			appendQuotedString(&output, name);
			output.charcat(':');
			internalSpacing();
			}

		if (quoted)
			appendQuotedString(&output, value);
		else
			output.memcat(value, valueLength);

		return true;
		}

public:

	void preFormattedValue(const char *name, const char *value)
		{
		internalPrint(name, value);
		}

	void string(const char *name, const char *value)
		{
		internalPrint(name, value, -1, true);
		}

	void integer(const char *name, int integer)
		{
		char buffer[64];
		sprintf(buffer, "%d", integer);
		internalPrint(name, buffer);
		}

	void integer64(const char *name, __int64 integer64)
		{
		char buffer[128];
		sprintf(buffer, "%I64d", integer);
		internalPrint(name, buffer);
		}

	void floatingPoint(const char *name, double floatingPoint)
		{
		char buffer[64];
		sprintf(buffer, "%f", floatingPoint);
		internalPrint(name, buffer);
		}

	void boolean(const char *name, bool boolean)
		{
		internalPrint(name, boolean ? "true" : "false");
		}

	void null(const char *name)
		{
		internalPrint(name, "null");
		}

	void binary(const char *name, const void *value, int length)
		{
		internalPrint(name, "binary");
		internalSpacing();
		char buffer[64];
		sprintf(buffer, "%d", length);
		output.strcat(buffer);
		internalSpacing();
		output.strcat(":");
		output.memcat((const char *)value, length);
		}

	void lineComment(const char *comment)
		{
		bool haveNewlines = (style == jsonfsTabsAndNewlines);
		// if we can't use newlines, we'll need to use a block
		// comment.  if block comments are disabled, don't print
		// the comment at all.
		if (!haveNewlines && (blockCommentStartString == NULL))
			return;
		internalPrint(NULL, (haveNewlines ? lineCommentString : blockCommentStartString), -1, false, false);
		char *commentStart = output;
		output.strcat(comment);

		// convert all eol characters into spaces
		// (nothing gets past me, mac.)
		char *trace;
		trace = commentStart;
		while ((trace = strchr(trace, '\n')) != NULL)
			*trace = ' ';
		trace = commentStart;
		while ((trace = strchr(trace, '\r')) != NULL)
			*trace = ' ';

		if (!haveNewlines)
			output.strcat(blockCommentEndString);
		wantComma = false;
		}

	void blockComment(const char *comment)
		{
		if (blockCommentStartString == NULL)
			return;
		if (wantComma)
			{
			output.charcat(',');
			wantComma = false;
			}
		if (!inComment)
			{
			startLine();
			output.strcat(blockCommentStartString);
			}
		inComment = true;
		startLine();
		if (style == jsonfsTabsAndNewlines)
			output.strcat(blockCommentLineStartString);
		output.strcat(comment);
		}

	void objectEnter(const char *name)
		{
		internalPrint(name, "{");
		genericEnter(jsonptObjectEnter);
		wantComma = false;
		}

	void arrayEnter(const char *name)
		{
		internalPrint(name, "[");
		genericEnter(jsonptArrayEnter);
		wantComma = false;
		}

	void objectLeave(void)
		{
		startLine();
		output.strcat("}");
		genericLeave(jsonptObjectEnter);
		}

	void arrayLeave(void) 
		{
		startLine();
		output.strcat("]");
		genericLeave(jsonptArrayEnter);
		}

	void finish(void)
		{
		while (state != NULL)
			{
			if (state->type == jsonptObjectEnter)
				objectLeave();
			else
				arrayLeave();
			}
		}
	};


struct jsonADT
	{
	jsonParserType type;
	dynamicString name;
	dynamicString stringOrBinary;
	int integer;
	__int64 integer64;
	double floatingPoint;
	bool boolean;
	jsonADT **children;
	int childrenCount; // both the size of "children" and the number contained within
	bool sortChildren;

	void setName(const char *name)
		{
		if (name != NULL)
			this->name.strcpy(name);
		}

	void setString(const char *name, const char *value)
		{
		setName(name);
		type = jsonptString;
		this->stringOrBinary.strcpy(value);
		}

	void setInteger(const char *name, int value)
		{
		setName(name);
		type = jsonptInteger;
		this->integer = value;
		}

	void setInteger64(const char *name, __int64 value)
		{
		setName(name);
		type = jsonptInteger64;
		this->integer64 = value;
		}

	void setFloatingPoint(const char *name, double value)
		{
		setName(name);
		type = jsonptFloatingPoint;
		this->floatingPoint = value;
		}

	void setBoolean(const char *name, bool value)
		{
		setName(name);
		type = jsonptBoolean;
		this->boolean = value;
		}

	void setNull(const char *name)
		{
		setName(name);
		type = jsonptNull;
		}

	void setBinary(const char *name, const void *value, int length)
		{
		setName(name);
		type = jsonptBinary;
		this->stringOrBinary.memcpy((const char *)value, length);
		}


private:

#ifndef MIN
#define MIN(a, b) ( ((a) < (b)) ? (a) : (b) )
#endif // MIN

	void internalAppend(jsonADT *child)
		{
		int index = childrenCount;
		childrenCount++;
		children = (jsonADT **)realloc(children, childrenCount * sizeof(jsonADT *));
		children[index] = child;

		// bubble-sort
		if (!sortChildren || (childrenCount < 2))
			return;

		bool changed = true;
		while (changed)
			{
			changed = false;
			for (int i = childrenCount - 2; i >= 0; i--)
				{
				int comparison;
				jsonADT *thisChild = children[i];
				jsonADT *nextChild = children[i + 1];
				if (thisChild->name.length && children[i + 1]->name.length)
					comparison = stricmp(thisChild->name, children[i + 1]->name);
				else if (thisChild->type != children[i + 1]->type)
					return;
				else
					{
					switch (thisChild->type)
						{
						case jsonptString:
							comparison = stricmp(thisChild->stringOrBinary, nextChild->stringOrBinary);
							break;
						case jsonptBinary:
							comparison = memcmp(thisChild->stringOrBinary, nextChild->stringOrBinary,
								MIN(thisChild->stringOrBinary.length, nextChild->stringOrBinary.length));
							break;
						case jsonptInteger:
							comparison = thisChild->integer - nextChild->integer;
							break;
						case jsonptInteger64:
							comparison = (int)(thisChild->integer64 - nextChild->integer64);
							break;
						case jsonptNull:
							return;
						case jsonptFloatingPoint:
							{
							double fcomparison = thisChild->floatingPoint - nextChild->floatingPoint;
							if (fcomparison < 0)
								comparison = -1;
							else if (fcomparison > 0)
								comparison = 1;
							else
								comparison = 0;
							break;
							}
						case jsonptBoolean:
							{
							// sort false to the front, I guess
							if (thisChild->boolean && !nextChild->boolean)
								comparison = -1;
							else if (!thisChild->boolean && nextChild->boolean)
								comparison = 1;
							else
								comparison = 0;
							break;
							}
						}
					}

				if (comparison > 0)
					{
					// swap
					children[i] = nextChild;
					children[i + 1] = thisChild;
					changed = true;
					}
				}
			}
		}
public:


	void setArray(const char *name)
		{
		setName(name);
		type = jsonptArrayEnter;
		}

	void arrayAppend(jsonADT *child)
		{
		if (child->name.length != 0)
			return;
		internalAppend(child);
		}

	void setObject(const char *name)
		{
		setName(name);
		type = jsonptObjectEnter;
		sortChildren = true;
		}

	void objectAppend(jsonADT *child)
		{
		if (child->name.length == 0)
			return;
		internalAppend(child);
		}


private:
	void initialize(void)
		{
		children = NULL;
		childrenCount = 0;
		sortChildren = false;
		}

	void recursiveConstructor(jsonParser *parser)
		{
		for (;;)
			{
			switch (parser->next())
				{
				case jsonEOF:
					return;

				case jsonptObjectLeave:
					assert(type == jsonptObjectEnter);
					return;

				case jsonptArrayLeave:
					assert(type == jsonptArrayEnter);
					return;

				default:
					{
					jsonADT *child = new jsonADT;
					switch (parser->type)
						{
						case jsonptString:
							child->setString(parser->name, parser->stringOrBinary);
							break;
						case jsonptBinary:
							child->setBinary(parser->name, parser->stringOrBinary, parser->stringOrBinary.length);
							break;
						case jsonptInteger:
							child->setInteger(parser->name, parser->integer);
							break;
						case jsonptInteger64:
							child->setInteger64(parser->name, parser->integer64);
							break;
						case jsonptFloatingPoint:
							child->setFloatingPoint(parser->name, parser->floatingPoint);
							break;
						case jsonptBoolean:
							child->setBoolean(parser->name, parser->boolean);
							break;
						case jsonptNull:
							child->setNull(parser->name);
							break;
						case jsonptObjectEnter:
							child->setObject(parser->name);
							child->recursiveConstructor(parser);
							break;
						case jsonptArrayEnter:
							child->setArray(parser->name);
							child->recursiveConstructor(parser);
							break;
						}

					if (type == jsonptObjectEnter)
						objectAppend(child);
					else
						arrayAppend(child);
					}
				}
			}
		}

public:

	jsonADT(void)
		{
		initialize();
		}


	jsonADT(jsonParser *parser)
		{
		initialize();
		type = parser->next();
		assert(type == jsonptObjectEnter);
		sortChildren = true;
		recursiveConstructor(parser);
		}


	void dump(jsonFormatter *f)
		{
		char *dumpName;
		if (name.length)
			dumpName = name;
		else
			dumpName = NULL;

		switch (type)
			{
			case jsonptString:
				f->string(dumpName, stringOrBinary);
				break;
			case jsonptBinary:
				f->binary(dumpName, stringOrBinary, stringOrBinary.length);
				break;
			case jsonptInteger:
				f->integer(dumpName, integer);
				break;
			case jsonptInteger64:
				f->integer64(dumpName, integer64);
				break;
			case jsonptFloatingPoint:
				f->floatingPoint(dumpName, floatingPoint);
				break;
			case jsonptBoolean:
				f->boolean(dumpName, boolean);
				break;
			case jsonptNull:
				f->null(dumpName);
				break;
			case jsonptObjectEnter:
				{
				f->objectEnter(dumpName);
				for (int i = 0; i < childrenCount; i++)
					children[i]->dump(f);
				f->objectLeave();
				break;
				}
			case jsonptArrayEnter:
				{
				f->arrayEnter(dumpName);
				for (int i = 0; i < childrenCount; i++)
					children[i]->dump(f);
				f->arrayLeave();
				break;
				}
			}
		}
	};


//
// JSON other externals
//

#if 0
char *test1 =
	" { "
	" \"name\" : \"value\", "
	" \"array\": [ "
	"	\"a\", "
	"	\"b\", "
	"	3 "
	"	], "
	" \"map\" :{ "
	"	\"x\" : true,"
	"	#test line comment\n"
	"	\"y\":false,"
	"	\"\\u004cch\":null, "
	"	\"long string\" : \"0123456789abcdef0123456789abcdef0123456789abcdef0123456789abcdef\","
	"	\"binary test\" : binary 5 :abc\td, "
	"	\"nested array\":["
	"		1,"
	"		2,"
	"		3,"
	"		], "
	"	\"nested map\":{"
	"		\"this\":\"that\""
	"		}, "
	"	\"goodbye!\" : \"l8r, sk8r\", "
	"	} ,"
	" } ";

char *test2 = 
"{\n"
"	\"glossary\": {\n"
"		\"title\": \"example glossary\",\n"
"		\"GlossDiv\": {\n"
"			\"title\": \"S\",\n"
"			\"GlossList\": [{\n"
"                \"ID\": \"SGML\",\n"
"                \"SortAs\": \"SGML\",\n"
"                \"GlossTerm\": \"Standard Generalized Markup Language\",\n"
"                \"Acronym\": \"SGML\",\n"
"                \"Abbrev\": \"ISO 8879:1986\",\n"
"                \"GlossDef\": \n"
"\"A meta-markup language, used to create markup languages such as DocBook.\",\n"
"                \"GlossSeeAlso\": [\"GML\", \"XML\", \"markup\"]\n"
"            }]\n"
"        }\n"
"    }\n"
"}\n"
;

char *test3 = 
"{\"menu\": {\n"
"  \"id\": \"file\",\n"
"  \"value\": \"File:\",\n"
"  \"popup\": {\n"
"    \"menuitem\": [\n"
"      {\"value\": \"New\", \"onclick\": \"CreateNewDoc()\"},\n"
"      {\"value\": \"Open\", \"onclick\": \"OpenDoc()\"},\n"
"      {\"value\": \"Close\", \"onclick\": \"CloseDoc()\"}\n"
"    ]\n"
"  }\n"
"}}\n"
;

char *test4 =
"{\"widget\": {\n"
"    \"debug\": \"on\",\n"
"    \"window\": {\n"
"        \"title\": \"Sample Konfabulator Widget\",\n"
"        \"name\": \"main_window\",\n"
"        \"width\": 500,\n"
"        \"height\": 500\n"
"    },\n"
"    \"image\": { \n"
"        \"src\": \"Images/Sun.png\",\n"
"        \"name\": \"sun1\",\n"
"        \"hOffset\": 250,\n"
"        \"vOffset\": 250,\n"
"        \"alignment\": \"center\"\n"
"    },\n"
"    \"text\": {\n"
"        \"data\": \"Click Here\",\n"
"        \"size\": 36,\n"
"        \"style\": \"bold\",\n"
"        \"name\": \"text1\",\n"
"        \"hOffset\": 250,\n"
"        \"vOffset\": 100,\n"
"        \"alignment\": \"center\",\n"
"        \"onMouseUp\": \"sun1.opacity = (sun1.opacity / 100) * 90;\"\n"
"    }\n"
"}}    \n"
;

char *test5 = 
"{\"menu\": {\n"
"    \"header\": \"SVG Viewer\",\n"
"    \"items\": [\n"
"        {\"id\": \"Open\"},\n"
"        {\"id\": \"OpenNew\", \"label\": \"Open New\"},\n"
"        null,\n"
"        {\"id\": \"ZoomIn\", \"label\": \"Zoom In\"},\n"
"        {\"id\": \"ZoomOut\", \"label\": \"Zoom Out\"},\n"
"        {\"id\": \"OriginalView\", \"label\": \"Original View\"},\n"
"        null,\n"
"        {\"id\": \"Quality\"},\n"
"        {\"id\": \"Pause\"},\n"
"        {\"id\": \"Mute\"},\n"
"        null,\n"
"        {\"id\": \"Find\", \"label\": \"Find...\"},\n"
"        {\"id\": \"FindAgain\", \"label\": \"Find Again\"},\n"
"        {\"id\": \"Copy\"},\n"
"        {\"id\": \"CopyAgain\", \"label\": \"Copy Again\"},\n"
"        {\"id\": \"CopySVG\", \"label\": \"Copy SVG\"},\n"
"        {\"id\": \"ViewSVG\", \"label\": \"View SVG\"},\n"
"        {\"id\": \"ViewSource\", \"label\": \"View Source\"},\n"
"        {\"id\": \"SaveAs\", \"label\": \"Save As\"},\n"
"        null,\n"
"        {\"id\": \"Help\"},\n"
"        {\"id\": \"About\", \"label\": \"About Adobe CVG Viewer...\"}\n"
"    ]\n"
"}}\n"
;

char *test6 =
"{\"web-app\": {\n"
"  \"servlet\": [    // Defines the CDSServlet\n"
"    {\n"
"      \"servlet-name\": \"cofaxCDS\",\n"
"      \"servlet-class\": \"org.cofax.cds.CDSServlet\",\n"
"/*\n"
"    Defines glossary variables that template designers\n"
"    can use across the site.  You can add new\n"
"    variables to this set by creating a new init-param, with\n"
"    the param-name prefixed with \"configGlossary:\".\n"
"*/\n"
"      \"init-param\": {\n"
"        \"configGlossary:installationAt\": \"Philadelphia, PA\",\n"
"        \"configGlossary:adminEmail\": \"ksm@pobox.com\",\n"
"        \"configGlossary:poweredBy\": \"Cofax\",\n"
"        \"configGlossary:poweredByIcon\": \"/images/cofax.gif\",\n"
"        \"configGlossary:staticPath\": \"/content/static\",\n"
"/*\n"
"    Defines the template loader and template processor\n"
"    classes.  These are implementations of org.cofax.TemplateProcessor\n"
"    and org.cofax.TemplateLoader respectively.  Simply create new\n"
"    implementation of these classes and set them here if the default\n"
"    implementations do not suit your needs.  Leave these alone\n"
"    for the defaults.\n"
"*/\n"
"        \"templateProcessorClass\": \"org.cofax.WysiwygTemplate\",\n"
"        \"templateLoaderClass\": \"org.cofax.FilesTemplateLoader\",\n"
"        \"templatePath\": \"templates\",\n"
"        \"templateOverridePath\": \"\",\n"
"/*\n"
"    Defines the names of the default templates to look for\n"
"    when acquiring WYSIWYG templates.  Leave these at their\n"
"    defaults for most usage.\n"
"*/\n"
"        \"defaultListTemplate\": \"listTemplate.htm\",\n"
"        \"defaultFileTemplate\": \"articleTemplate.htm\",\n"
"/*\n"
"    New! useJSP switches on JSP template processing.\n"
"    jspListTemplate and jspFileTemplate are the names\n"
"    of the default templates to look for when aquiring JSP\n"
"    templates.  Cofax currently in production at KR has useJSP\n"
"    set to false, since our sites currently use WYSIWYG\n"
"    templating exclusively.\n"
"*/\n"
"        \"useJSP\": false,\n"
"        \"jspListTemplate\": \"listTemplate.jsp\",\n"
"        \"jspFileTemplate\": \"articleTemplate.jsp\",\n"
"/*\n"
"    Defines the packageTag cache.  This cache keeps\n"
"    Cofax from needing to interact with the database\n"
"    to look up packageTag commands.\n"
"*/\n"
"        \"cachePackageTagsTrack\": 200,\n"
"        \"cachePackageTagsStore\": 200,\n"
"        \"cachePackageTagsRefresh\": 60,\n"
"/*\n"
"    Defines the template cache.  Keeps Cofax from needing\n"
"    to go to the file system to load a raw template from\n"
"    the file system.\n"
"*/\n"
"        \"cacheTemplatesTrack\": 100,\n"
"        \"cacheTemplatesStore\": 50,\n"
"        \"cacheTemplatesRefresh\": 15,\n"
"/*\n"
"    Defines the page cache.  Keeps Cofax from processing\n"
"    templates to deliver to users.\n"
"*/\n"
"        \"cachePagesTrack\": 200,\n"
"        \"cachePagesStore\": 100,\n"
"        \"cachePagesRefresh\": 10,\n"
"        \"cachePagesDirtyRead\": 10,\n"
"/*\n"
"    Defines the templates Cofax will use when\n"
"    being browsed by a search engine identified in\n"
"    searchEngineRobotsDb\n"
"*/\n"
"        \"searchEngineListTemplate\": \"forSearchEnginesList.htm\",\n"
"        \"searchEngineFileTemplate\": \"forSearchEngines.htm\",\n"
"        \"searchEngineRobotsDb\": \"WEB-INF/robots.db\",\n"
"/*\n"
"    New!  useDataStore enables/disables the Cofax database pool\n"
"*/\n"
"        \"useDataStore\": true,\n"
"/*\n"
"    Defines the implementation of org.cofax.DataStore that Cofax\n"
"    will use.  If this DataStore class does not suit your needs\n"
"    simply implement a new DataStore class and set here.\n"
"*/\n"
"        \"dataStoreClass\": \"org.cofax.SqlDataStore\",\n"
"/*\n"
"    Defines the implementation of org.cofax.Redirection that\n"
"    Cofax will use.  If this Redirection class does not suit\n"
"    your needs simply implenet a new Redirection class\n"
"    and set here.\n"
"*/\n"
"        \"redirectionClass\": \"org.cofax.SqlRedirection\",\n"
"/*\n"
"    Defines the data store name.   Keep this at the default\n"
"*/\n"
"        \"dataStoreName\": \"cofax\",\n"
"/*\n"
"    Defines the JDBC driver that Cofax's database pool will use\n"
"*/\n"
"        \"dataStoreDriver\": \"com.microsoft.jdbc.sqlserver.SQLServerDriver\",\n"
"/*\n"
"    Defines the JDBC connection URL to connect to the database\n"
"*/\n"
"        \"dataStoreUrl\": \"jdbc:microsoft:sqlserver://LOCALHOST:1433;DatabaseName=goon\",\n"
"/*\n"
"    Defines the user name to connect to the database\n"
"*/\n"
"        \"dataStoreUser\": \"sa\",\n"
"/*\n"
"    Defines the password to connect to the database\n"
"*/\n"
"        \"dataStorePassword\": \"dataStoreTestQuery\",\n"
"/*\n"
"    A query that will run to test the validity of the\n"
"    connection in the pool.\n"
"*/\n"
"        \"dataStoreTestQuery\": \"SET NOCOUNT ON;select test='test';\",\n"
"/*\n"
"    A log file to print out database information\n"
"*/\n"
"        \"dataStoreLogFile\": \"/usr/local/tomcat/logs/datastore.log\",\n"
"/*\n"
"    The number of connection to initialize on startup\n"
"*/\n"
"        \"dataStoreInitConns\": 10,\n"
"/*\n"
"    The maximum number of connection to use in the pool\n"
"*/\n"
"        \"dataStoreMaxConns\": 100,\n"
"/*\n"
"    The number of times a connection will be utilized from the\n"
"    pool before disconnect\n"
"*/\n"
"        \"dataStoreConnUsageLimit\": 100,\n"
"/*\n"
"    The level of information to print to the log\n"
"*/\n"
"        \"dataStoreLogLevel\": \"debug\",\n"
"/*\n"
"    The maximum URL length allowable by the CDS Servlet\n"
"    Helps to prevent hacking\n"
"*/\n"
"        \"maxUrlLength\": 500}},\n"
"/*\n"
"    Defines the Email Servlet\n"
"*/\n"
"    {\n"
"      \"servlet-name\": \"cofaxEmail\",\n"
"      \"servlet-class\": \"org.cofax.cds.EmailServlet\",\n"
"      \"init-param\": {\n"
"/*\n"
"    The mail host to be used by the mail servlet\n"
"*/\n"
"        \"mailHost\": \"mail1\",\n"
"/*\n"
"    An override\n"
"*/\n"
"        \"mailHostOverride\": \"mail2\"}},\n"
"/*\n"
"    Defines the Admin Servlet - used to refresh cache on\n"
"    demand and see statistics\n"
"*/\n"
"    {\n"
"      \"servlet-name\": \"cofaxAdmin\",\n"
"      \"servlet-class\": \"org.cofax.cds.AdminServlet\"},\n"
"/*\n"
"    Defines the File Servlet - used to display files like Apache\n"
"*/\n"
"    {\n"
"      \"servlet-name\": \"fileServlet\",\n"
"      \"servlet-class\": \"org.cofax.cds.FileServlet\"},\n"
"    {\n"
"      \"servlet-name\": \"cofaxTools\",\n"
"      \"servlet-class\": \"org.cofax.cms.CofaxToolsServlet\",\n"
"      \"init-param\": {\n"
"/*\n"
"    Path to the template folder relative to the tools tomcat installation.\n"
"*/\n"
"        \"templatePath\": \"toolstemplates/\",\n"
"/*\n"
"    Logging boolean 1 = on, 0 = off\n"
"*/\n"
"        \"log\": 1,\n"
"/*\n"
"    Location of log. If empty, log will be written System.out\n"
"*/\n"
"        \"logLocation\": \"/usr/local/tomcat/logs/CofaxTools.log\",\n"
"/*\n"
"    Max size of log in BITS. If size is empty, no limit to log.\n"
"    If size is defined, log will be overwritten upon reaching defined size.\n"
"*/\n"
"        \"logMaxSize\": \"\",\n"
"/*\n"
"    DataStore logging boolean 1 = on, 0 = off\n"
"*/\n"
"        \"dataLog\": 1,\n"
"/*\n"
"    DataStore location of log. If empty, log will be written System.out\n"
"*/\n"
"        \"dataLogLocation\": \"/usr/local/tomcat/logs/dataLog.log\",\n"
"/*\n"
"    Max size of log in BITS. If size is empty, no limit to log.\n"
"    If size is defined, log will be overwritten upon reaching defined size.\n"
"*/\n"
"        \"dataLogMaxSize\": \"\",\n"
"/*\n"
"    Http string relative to server root to call for page cache\n"
"    removal to Cofax Servlet.\n"
"*/\n"
"        \"removePageCache\": \"/content/admin/remove?cache=pages&id=\",\n"
"/*\n"
"    Http string relative to server root to call for template\n"
"    cache removal to Cofax Servlet.\n"
"*/\n"
"        \"removeTemplateCache\": \"/content/admin/remove?cache=templates&id=\",\n"
"/*\n"
"    Location of folder from root of drive that will be used for\n"
"    ftp transfer from beta server or user hard drive to live servers.\n"
"    Note that Edit Article will not function without this variable\n"
"    set correctly. MultiPart request relies upon access to this folder.\n"
"*/\n"
"        \"fileTransferFolder\": \"/usr/local/tomcat/webapps/content/fileTransferFolder\",\n"
"/*\n"
"    Defines whether the Server should look in another path for\n"
"    config files or variables.\n"
"*/\n"
"        \"lookInContext\": 1,\n"
"/*\n"
"    Number of the ID of the top level administration group in tblPermGroups.\n"
"*/\n"
"        \"adminGroupID\": 4,\n"
"/*\n"
"    Is the tools app running on  the 'beta server'.\n"
"*/\n"
"        \"betaServer\": true}}],\n"
"  \"servlet-mapping\": {\n"
"/*\n"
"    URL mapping for the CDS Servlet\n"
"*/\n"
"     \"cofaxCDS\": \"/\",\n"
"/*\n"
"    URL mapping for the Email Servlet\n"
"*/\n"
"     \"cofaxEmail\": \"/cofaxutil/aemail/*\",\n"
"/*\n"
"    URL mapping for the Admin servlet\n"
"*/\n"
"     \"cofaxAdmin\": \"/admin/*\",\n"
"/*\n"
"    URL mapping for the Files servlet\n"
"*/\n"
"     \"fileServlet\": \"/static/*\",\n"
"     \"cofaxTools\": \"/tools/*\"},\n"
"/*\n"
"    New! The cofax taglib descriptor file\n"
"*/\n"
"  \"taglib\": {\n"
"    \"taglib-uri\": \"cofax.tld\",\n"
"    \"taglib-location\": \"/WEB-INF/tlds/cofax.tld\"}}}\n"
;

char *test = test1;

#endif // 0

static char nesting[] = "++++++++++++++++++++++++++++++++++++++++";
static char *nestingTail = nesting + sizeof(nesting) - 1;


void dumpParser(const char *testString)
	{
	jsonParser jp(testString);
	for (;;)
		{
		jsonParserType t = jp.next();

		if (t == jsonEOF)
			{
			printf("<<eof>>\n");
			break;
			}

		printf("%s %11s (%3d) ",
			nestingTail - jp.state->level,
			jsonDescribeParserType(t),
			t);

		if (*jp.name)
			printf("%s = ", jp.name);

		switch (t)
			{
			case jsonptString:
				printf("\"%s\"", jp.stringOrBinary);
				break;
			case jsonptBinary:
				printf("%d binary bytes (\"%s\")", jp.stringOrBinaryLength, jp.stringOrBinary);
				break;
			case jsonptInteger:
				printf("%d", jp.integer);
				break;
			case jsonptInteger64:
				printf("%I64d (i64)", jp.integer64);
				break;
			case jsonptBoolean:
				printf("%s", jp.boolean ? "true" : "false");
				break;
			case jsonptFloatingPoint:
				printf("%f", jp.floatingPoint);
				break;
			case jsonptObjectEnter:
				printf("{");
				break;
			case jsonptObjectLeave:
				printf("}");
				break;
			case jsonptArrayEnter:
				printf("[");
				break;
			case jsonptArrayLeave:
				printf("]");
				break;
			}
		printf("\n");
		}
	}

int main(int argc, char *argv[])
	{
	char *test = "{ \"a\" : 5 }";
	char *trace = test;
	jsonScanner js(test);
	for (;;)
		{
		jsonToken t = js.next();
		printf("%11s (%3d)", jsonDescribeToken(t), t);
		if (t == jsonBinary)
			printf(" (%d bytes)", js.length);
		else if (*js.buffer)
			printf(" \"%s\"", js.buffer);
		printf("\n");
		if (t == jsonEOF)
			break;
		}

	dumpParser(test);

//	jsonFormatter f(jsonfsNoWhitespace);
//	jsonFormatter f(jsonfsOneSpace);
	jsonFormatter f;

	f.string("name", "value");
	f.arrayEnter("array");
		f.string(NULL, "a");
		f.string(NULL, "b\nnewline!");
		f.integer(NULL, 3);
	f.arrayLeave();
	f.objectEnter("map");
		f.boolean("x", true);
		f.lineComment("test line comment");
		f.boolean("y", false);
		f.string("long string", "0123456789abcdef0123456789abcdef0123456789abcdef0123456789abcdef");
		f.binary("binary test", "abc" "\xdc" "d", 5);
		f.arrayEnter("nestedArray");
			f.integer(NULL, 1);
			f.integer(NULL, 2);
			f.integer(NULL, 3);
		f.arrayLeave();
		f.objectEnter("map");
			f.string("this", "that");
		f.objectLeave();
		f.string("quoteables", "ab\t\xef");
		f.string("goodbye!", "l8r, sk8r");
		f.blockComment("first line, I guess.");
		f.blockComment("second line, hooray.");
		f.string("okay, I guess", "it wasn't goodbye after all");
		f.string("gonna stop now", "no more strings for you");
		f.binary("embedded zero", "321\0!!!", 7);
		f.string("after zero", "you shouldn't see this.");
	f.objectLeave();

	f.finish();
	printf("formatted f:\n%s\n", (const char *)f.output);

	dumpParser(f.output);

	jsonParser testParser(test);
	jsonADT adt(&testParser);
	jsonFormatter formatter;
	adt.dump(&formatter);
	puts(formatter.output);


	return 0;
	}


