/*
** [BEGIN NOTICE]
**
** Copyright (C) 1998 Larry Hastings
**
** This software is provided 'as-is', without any express or implied warranty.
** In no event will the authors be held liable for any damages arising from
** the use of this software.
**
** Permission is granted to anyone to use this software for any purpose,
** including commercial applications, and to alter it and redistribute
** it freely, subject to the following restrictions:
**
** 1. The origin of this software must not be misrepresented; you must not
**    claim that you wrote the original software. If you use this software
**    in a product, an acknowledgment in the product documentation would be
**    appreciated but is not required.
** 2. Altered source versions must be plainly marked as such, and must not be
**    misrepresented as being the original software.
** 3. This notice may not be removed or altered from any source distribution.
**
** The Fiji homepage is here:
**		http://www.midwinter.com/~lch/programming/fiji/
**
** [END NOTICE]
*/

import Fiji.*;
import java.io.*;
import java.lang.*;
import java.util.Hashtable;
import java.util.StringTokenizer;
import java.util.Vector;


public class FijiInterpreter extends Object
	{
	// program array and counters
	public FijiPointer program;
	public FijiDictionary dictionary;
	static public FijiDictionary globalDictionary;

	boolean noByeYet = true;
	public boolean bye() 
		{
		return !noByeYet;
		}

	static FijiPointer doVariable = new FijiPointer();
	static FijiPointer doLocal;
	static FijiPointer doColon = new FijiDoColon();
	static FijiPointer doDoes = new FijiDoDoes();

	// data stack
	Vector DS = new Vector();
	// return stack
	Vector RS = new Vector();
	// control stack
	Vector CS = new Vector();
	// locals stack
	Vector LS = new Vector();

	boolean creating = false;
	boolean compiling = false;
	FijiWord newWord;
	boolean postponeNext;
	FijiWord postponedWord;
	FijiPointer doesProgram;
	Vector	bytes = null;
	Vector	objects = null;

	// bytecodes, produced by "dollar fijibytes.$"






	// types, also double as bytecodes
	static final byte INVALID = -127;
	static final byte STRING = -126;
	static final byte POINTER = -125;
	static final byte METHOD = -124;
	static final byte DOUBLE = -123;
	static final byte FLOAT = -122;
	static final byte LONG = -121;
	static final byte INTEGER = -120;
	static final byte SHORT = -119;
	static final byte BYTE = -118;
	static final byte CHARACTER = -117;
	static final byte BOOLEAN = -116;

	// conversions
	static final byte TO_CLASS = -115;
	static final byte TO_BODY = -114;
	static final byte TO_CFA = -113;

	// flow control
	static final byte CALL = -112;
	static final byte EXIT = -111;
	static final byte EXECUTE = -110;
	static final byte INTERNAL_STOP = -109;
	static final byte JUMP = -108;
	static final byte JUMP_CHECK = -107;
	static final byte BRANCH0 = -106;
	static final byte BRANCH0_CHECK = -105;
	static final byte JUMP_BACKWARD_PUSH = -104;
	static final byte JUMP_FORWARD_PUSH = -103;
	static final byte JUMP_BACKWARD_FIXUP = -102;
	static final byte JUMP_FORWARD_FIXUP = -101;
	static final byte CS_ROLL = -100;
	static final byte CS_PICK = -99;
	static final byte TO_CS = -98;
	static final byte CS_FROM = -97;

	// keep LEAVE, DO, and QUESTION_DO _after_ JUMP_FORWARD_PUSH (see pushFixup())
	static final byte DO = -96;
	static final byte QUESTION_DO = -95;
	static final byte I = -94;
	static final byte J = -93;
	static final byte K = -92;
	static final byte LEAVE = -91;
	static final byte I_LEAVE = -90;
	static final byte I_LEAVE_CHECK = -89;

	// I_LOOP must = LOOP + 1, I_LOOP_CHECK must = LOOP + 2, same for I_PLUS_LOOP
	static final byte LOOP = -88;
	static final byte I_LOOP = -87;
	static final byte I_LOOP_CHECK = -86;
	static final byte PLUS_LOOP = -85;
	static final byte I_PLUS_LOOP = -84;
	static final byte I_PLUS_LOOP_CHECK = -83;
	static final byte I_PREP_DO = -82;
	static final byte I_DO = -81;
	static final byte I_DO_CHECK = -80;
	static final byte I_QUESTION_DO = -79;
	static final byte I_QUESTION_DO_CHECK = -78;

	// math operands
	static final byte ADD = -77;
	static final byte SUBTRACT = -76;
	static final byte MULTIPLY = -75;
	static final byte DIVIDE = -74;
	static final byte MOD = -73;
	static final byte SHIFT_LEFT = -72;
	static final byte SHIFT_RIGHT = -71;
	static final byte SHIFT_RIGHT0 = -70;
	static final byte BINARY_AND = -69;
	static final byte BINARY_OR = -68;
	static final byte BINARY_XOR = -67;
	static final byte BOOLEAN_AND = -66;
	static final byte BOOLEAN_OR = -65;
	static final byte BOOLEAN_XOR = -64;
	static final byte LESS_THAN = -63;
	static final byte LESS_THAN_EQUALS = -62;
	static final byte EQUALS = -61;
	static final byte GREATER_THAN_EQUALS = -60;
	static final byte GREATER_THAN = -59;
	static final byte NOT_EQUAL_TO = -58;
	static final byte NOT = -57;

	// stack operations
	static final byte DROP = -56;
	static final byte DUP = -55;
	static final byte QUESTION_DUP = -54;
	static final byte SWAP = -53;
	static final byte ROT = -52;
	static final byte PUSH = -51;
	static final byte LITERAL = -50;
	static final byte ROLL = -49;
	static final byte PICK = -48;
	static final byte DEPTH = -47;
	static final byte TO_R = -46;
	static final byte R_FROM = -45;
	static final byte R_FETCH = -44;

	// strings
	static final byte QUOTE = -43;
	static final byte DOT_QUOTE = -42;
	static final byte NEXT_TOKEN = -41;
	static final byte NEXT_NONWHITE_TOKEN = -40;
	static final byte MORE_TOKENS = -39;
	static final byte PUT_BACK_TOKEN = -38;
	static final byte STRICMP = -37;
	static final byte STRTOK = -36;
	static final byte SUBSTR = -35;
	static final byte POPCHAR = -34;

	// words
	static final byte ALLOT = -33;
	static final byte COMMA = -32;
	static final byte FETCH = -31;
	static final byte TO_LENGTH = -30;
	static final byte STORE = -29;
	static final byte C_ALLOT = -28;
	static final byte C_COMMA = -27;
	static final byte C_FETCH = -26;
	static final byte C_TO_LENGTH = -25;
	static final byte C_STORE = -24;
	static final byte C_PLUS = -23;
	static final byte C_MINUS = -22;
	static final byte BODY_STORE = -21;
	static final byte CFA_STORE = -20;
	static final byte DOES_SET_CFA = -19;

	// compiling words
	static final byte COLON = -18;
	static final byte COLON_PRIME = -17;
	static final byte SEMICOLON = -16;
	static final byte CREATE = -15;
	static final byte CREATE_PRIME = -14;
	static final byte FINALIZE = -13;
	static final byte LEFT_SQUARE = -12;
	static final byte RIGHT_SQUARE = -11;
	static final byte STATE_QUESTION = -10;
	static final byte POSTPONE = -9;
	static final byte HANDLE_POSTPONED_WORD = -8;
	static final byte HANDLE_POSTPONED_BYTECODE = -7;
	static final byte IMMEDIATE = -6;
	static final byte INLINE = -5;
	static final byte TICK = -4;
	static final byte TICK_PRIME = -3;
	static final byte COMPILE_BYTE = -2;
	static final byte COMPILE_OBJECT = 17;
	static final byte RECURSE = 18;
	static final byte DOES_GT = 19;

	// permutations of . and ?--keep DOT and QUESTION operands in same order!
	static final byte DOT = 20;
	static final byte DOT_DOT = 21;
	static final byte DOT_X = 22;
	static final byte DOT_DOT_X = 23;
	static final byte DOT_O = 24;
	static final byte DOT_DOT_O = 25;
	static final byte DOT_B = 26;
	static final byte DOT_DOT_B = 27;
	static final byte QUESTION = 28;
	static final byte QUESTION_QUESTION = 29;
	static final byte QUESTION_X = 30;
	static final byte QUESTION_QUESTION_X = 31;
	static final byte QUESTION_O = 32;
	static final byte QUESTION_QUESTION_O = 33;
	static final byte QUESTION_B = 34;
	static final byte QUESTION_QUESTION_B = 35;
	static final byte DOT_R = 36;

	// miscellaneous
	static final byte BYE = 37;
	static final byte CLOCK = 38;
	static final byte CR = 39;
	static final byte EMIT = 40;
	static final byte BACKSLASH = 41;
	static final byte SEE = 42;
	static final byte EVALUATE = 43;
	static final byte INCLUDE = 44;
	static final byte LEFT_PAREN = 45;
	static final byte ADD_PREFIX = 46;
	static final byte REMOVE_PREFIX = 47;
	static final byte REFLECT = 48;
	static final byte MOVE = 49;
	static final byte PUSH_FIJI_INTERPRETER = 50;
	static final byte HERE = 51;
	static final byte NULL = 52;
	static final byte TO_BYTEX = 53;
	static final byte TO_OBJECTX = 54;
	static final byte BYTEX_STORE = 55;
	static final byte OBJECTX_STORE = 56;
	static final byte TO_SERIALNUMBER = 57;
	static final byte BRACKET_TICK = 58;
	static final byte MOST_RECENT_WORD = 59;
	static final byte DICTIONARY_NEW = 60;
	static final byte DICTIONARY_FETCH = 61;
	static final byte DICTIONARY_STORE = 62;
	static final byte INTERNAL_LOCAL = 63;
	static final byte LS_PICK = 64;
	static final byte TO_LS = 65;
	static final byte LS_FROM = 66;
	static final byte LS_REPLACE = 67;
	static final byte REPLACE = 68;
	static final byte REFLECT_MAKE_CONSTRUCTOR = 69;
	static final byte REFLECT_MAKE_METHOD = 70;

	static Hashtable bytehash = null;
	static String[] byteToString = null;
	static boolean[] byteImmediate = null;
	static int[] byteArgumentCount = null;

	static void bytehashAdd(String s, byte parameter, int argumentCount)
		{
		bytehash.put(s.toLowerCase(), new Byte(parameter));
		byteToString[parameter + 128] = s;
		byteArgumentCount[parameter + 128] = argumentCount;
		}

	static void bytehashAdd(String s, byte parameter)
		{
		bytehashAdd(s, parameter, 0);
		}

	static void bytehashAddInteger(int parameter)
		{
		Integer x = new Integer(parameter);
		bytehashAdd(x.toString(), x.byteValue());
		}

	public static boolean getByteCodeImmediate(byte b)
		{
		if (byteImmediate == null)
			getByteCode(".");  // prime the pump
		return byteImmediate[((int)b) + 128];
		}

	public static void setByteCodeImmediate(byte b)
		{
		byteImmediate[((int)b) + 128] = true;
		}

	public static byte getByteCode(String s)
		{
		if (bytehash == null)
			{
			bytehash = new Hashtable();
			byteToString  = new String[256];
			byteImmediate = new boolean[256];
			byteArgumentCount = new int[256];

			bytehashAdd("not", NOT);

			bytehashAdd("+", ADD);
			bytehashAdd("-", SUBTRACT);
			bytehashAdd("*", MULTIPLY);
			bytehashAdd("/", DIVIDE);
			bytehashAdd("%", MOD);
			bytehashAdd("mod", MOD);
			bytehashAdd("<<", SHIFT_LEFT);
			bytehashAdd(">>", SHIFT_RIGHT);
			bytehashAdd(">>>", SHIFT_RIGHT0);

			bytehashAdd("&", BINARY_AND);
			bytehashAdd("|", BINARY_OR);
			bytehashAdd("^", BINARY_XOR);
			bytehashAdd("&&", BOOLEAN_AND);
			bytehashAdd("||", BOOLEAN_OR);
			bytehashAdd("^^", BOOLEAN_XOR);

			bytehashAdd("<", LESS_THAN);
			bytehashAdd("<=", LESS_THAN_EQUALS);
			bytehashAdd("=", EQUALS);
			bytehashAdd("==", EQUALS);
			bytehashAdd(">=", GREATER_THAN_EQUALS);
			bytehashAdd(">", GREATER_THAN);
			bytehashAdd("!=", NOT_EQUAL_TO);
			bytehashAdd("<>", NOT_EQUAL_TO);

			bytehashAdd(".", DOT);
			bytehashAdd(".x", DOT_X);
			bytehashAdd(".o", DOT_O);
			bytehashAdd(".b", DOT_B);
			bytehashAdd("..", DOT_DOT);
			bytehashAdd("..x", DOT_DOT_X);
			bytehashAdd("..o", DOT_DOT_O);
			bytehashAdd("..b", DOT_DOT_B);

			bytehashAdd("?", QUESTION);
			bytehashAdd("?x", QUESTION_X);
			bytehashAdd("?o", QUESTION_O);
			bytehashAdd("?b", QUESTION_B);
			bytehashAdd("??", QUESTION_QUESTION);
			bytehashAdd("??x", QUESTION_QUESTION_X);
			bytehashAdd("??o", QUESTION_QUESTION_O);
			bytehashAdd("??b", QUESTION_QUESTION_B);

			bytehashAdd("here", HERE);
			bytehashAdd("null", NULL);

			bytehashAdd(".r", DOT_R);


			bytehashAdd("emit", EMIT);
			bytehashAdd("cr", CR);

			bytehashAdd("\"", QUOTE, 1);
			setByteCodeImmediate(QUOTE);
			bytehashAdd(".\"", DOT_QUOTE, 1);
			setByteCodeImmediate(DOT_QUOTE);

			bytehashAdd("\\", BACKSLASH);
			bytehashAdd("//", BACKSLASH);
			setByteCodeImmediate(BACKSLASH);
			bytehashAdd("(", LEFT_PAREN);
			setByteCodeImmediate(LEFT_PAREN);

			bytehashAdd("swap", SWAP);
			bytehashAdd("rot", ROT);
			bytehashAdd("dup", DUP);
			bytehashAdd("?dup", QUESTION_DUP);
			bytehashAdd("drop", DROP);
			bytehashAdd("pick", PICK);
			bytehashAdd("replace", REPLACE);
			bytehashAdd("roll", ROLL);
			bytehashAdd(">r", TO_R);
			bytehashAdd("r>", R_FROM);
			bytehashAdd("r@", R_FETCH);

			bytehashAdd(":", COLON);
			setByteCodeImmediate(COLON);
			bytehashAdd(":'", COLON_PRIME);
			setByteCodeImmediate(COLON_PRIME);
			bytehashAdd(";", SEMICOLON);
			setByteCodeImmediate(SEMICOLON);
			bytehashAdd("recurse", RECURSE, 1);
			setByteCodeImmediate(RECURSE);

			bytehashAdd("[", LEFT_SQUARE);
			setByteCodeImmediate(LEFT_SQUARE);
			bytehashAdd("]", RIGHT_SQUARE);
			setByteCodeImmediate(RIGHT_SQUARE);
			bytehashAdd("literal", LITERAL);
			setByteCodeImmediate(LITERAL);
			bytehashAdd("push", PUSH, 1);
			bytehashAdd("execute", EXECUTE);

			bytehashAdd("'", TICK);
			bytehashAdd("''", TICK_PRIME);
			bytehashAdd("[']", BRACKET_TICK);
			setByteCodeImmediate(BRACKET_TICK);


			bytehashAdd("postpone", POSTPONE);
			setByteCodeImmediate(POSTPONE);
			bytehashAdd("compilebyte", COMPILE_BYTE);
			bytehashAdd("compileobject", COMPILE_OBJECT, 1);
			bytehashAdd(">body", TO_BODY);
			bytehashAdd(">cfa", TO_CFA);
			bytehashAdd("body!", BODY_STORE);
			bytehashAdd("cfa!", CFA_STORE);
			bytehashAdd(">bytex", TO_BYTEX);
			bytehashAdd(">objectx", TO_OBJECTX);
			bytehashAdd(">serialnumber", TO_SERIALNUMBER);
			bytehashAdd("bytex!", BYTEX_STORE);
			bytehashAdd("objectx!", OBJECTX_STORE);

			bytehashAdd("@", FETCH);
			bytehashAdd("!", STORE);

			bytehashAdd("nexttoken", NEXT_TOKEN);
			bytehashAdd("nextnonwhitetoken", NEXT_NONWHITE_TOKEN);
			bytehashAdd("moretokens", MORE_TOKENS);
			bytehashAdd("putbacktoken", PUT_BACK_TOKEN);
			bytehashAdd("stricmp", STRICMP);
			bytehashAdd("strtok", STRTOK);
			bytehashAdd("substr", SUBSTR);
			bytehashAdd("popchar", POPCHAR);

			bytehashAdd("state?", STATE_QUESTION);

			bytehashAdd("create", CREATE);
			bytehashAdd("create'", CREATE_PRIME);
			bytehashAdd("does>", DOES_GT);
			setByteCodeImmediate(DOES_GT);
			bytehashAdd(";;", FINALIZE);
			bytehashAdd("finalize", FINALIZE);
			bytehashAdd("immediate", IMMEDIATE);
			bytehashAdd("inline", INLINE);
			bytehashAdd(",", COMMA);
			bytehashAdd("c,", C_COMMA);
			bytehashAdd("allot", ALLOT);
			bytehashAdd("callot", C_ALLOT);
			bytehashAdd("addprefix", ADD_PREFIX);
			bytehashAdd("removeprefix", REMOVE_PREFIX);
			bytehashAdd("bye", BYE);

			bytehashAdd("c+", C_PLUS);
			bytehashAdd("c-", C_MINUS);

			// inspectors
			bytehashAdd(">length", TO_LENGTH);
			bytehashAdd("c>length", C_TO_LENGTH);
			bytehashAdd(">class", TO_CLASS);

			bytehashAdd("(call)", CALL, 1);
			bytehashAdd("exit", EXIT);
			bytehashAdd("(stop)", INTERNAL_STOP);
			bytehashAdd("(jump)", JUMP, 2);
			bytehashAdd("(jump-check)", JUMP_CHECK, 2);
			bytehashAdd("(branch0)", BRANCH0, 2);
			bytehashAdd("(branch0-check)", BRANCH0_CHECK, 2);
			bytehashAdd("(jump-forward-push)", JUMP_FORWARD_PUSH);
			bytehashAdd("(jump-backward-push)", JUMP_BACKWARD_PUSH);
			bytehashAdd("(jump-forward-fixup)", JUMP_FORWARD_FIXUP);
			bytehashAdd("(jump-backward-fixup)", JUMP_BACKWARD_FIXUP);
			bytehashAdd("(does-set-cfa)", DOES_SET_CFA, 1);
			bytehashAdd("(handle-postponed-word)", HANDLE_POSTPONED_WORD, 1);
			bytehashAdd("(handle-postponed-bytecode)", HANDLE_POSTPONED_BYTECODE);
			bytehashAdd("cs-roll", CS_ROLL);
			bytehashAdd("cs-pick", CS_PICK);

			bytehashAdd("do", DO);
			setByteCodeImmediate(DO);
			bytehashAdd("?do", QUESTION_DO);
			setByteCodeImmediate(QUESTION_DO);
			bytehashAdd("loop", LOOP);
			setByteCodeImmediate(LOOP);
			bytehashAdd("+loop", PLUS_LOOP);
			setByteCodeImmediate(PLUS_LOOP);
			bytehashAdd("leave", LEAVE);
			setByteCodeImmediate(LEAVE);
			bytehashAdd("i", I);
			bytehashAdd("j", J);
			bytehashAdd("k", K);
			bytehashAdd("(do)", I_DO, 2);
			bytehashAdd("(?do)", I_QUESTION_DO, 2);
			bytehashAdd("(do-check)", I_DO_CHECK, 2);
			bytehashAdd("(?do-check)", I_QUESTION_DO_CHECK, 2);
			bytehashAdd("(loop)", I_LOOP, 2);
			bytehashAdd("(+loop)", I_PLUS_LOOP, 2);
			bytehashAdd("(leave)", I_LEAVE, 2);
			bytehashAdd("(loop-check)", I_LOOP_CHECK, 2);
			bytehashAdd("(+loop-check)", I_PLUS_LOOP_CHECK, 2);
			bytehashAdd("(leave-check)", I_LEAVE_CHECK, 2);
			bytehashAdd("(prep-do)", I_PREP_DO);

			bytehashAdd("push-fiji-interpreter", PUSH_FIJI_INTERPRETER);
			bytehashAdd("most-recent-word", MOST_RECENT_WORD);
			bytehashAdd("dictionary-new", DICTIONARY_NEW);
			bytehashAdd("dictionary-store", DICTIONARY_STORE);
			bytehashAdd("dictionary-fetch", DICTIONARY_FETCH);
			bytehashAdd("(local)", INTERNAL_LOCAL);
			// setByteCodeImmediate(INTERNAL_LOCAL);
			bytehashAdd("ls-pick", LS_PICK);
			bytehashAdd("ls-replace", LS_REPLACE);
			bytehashAdd(">ls", TO_LS);
			bytehashAdd("ls>", LS_FROM);

			bytehashAdd("reflect", REFLECT);
			bytehashAdd("reflect-make-constructor", REFLECT_MAKE_CONSTRUCTOR);
			bytehashAdd("reflect-make-method", REFLECT_MAKE_METHOD);

			bytehashAdd("see", SEE);
			setByteCodeImmediate(SEE);
			bytehashAdd("depth", DEPTH);
			bytehashAdd("include", INCLUDE);
			bytehashAdd("evaluate", EVALUATE);

			bytehashAdd("clock", CLOCK);
			bytehashAdd("move", MOVE);

			bytehashAddInteger(-1);
			bytehashAddInteger(0);
			bytehashAddInteger(1);
			bytehashAddInteger(2);
			bytehashAddInteger(3);
			bytehashAddInteger(4);
			bytehashAddInteger(5);
			bytehashAddInteger(6);
			bytehashAddInteger(7);
			bytehashAddInteger(8);
			bytehashAddInteger(9);
			bytehashAddInteger(10);
			bytehashAddInteger(11);
			bytehashAddInteger(12);
			bytehashAddInteger(13);
			bytehashAddInteger(14);
			bytehashAddInteger(15);
			bytehashAddInteger(16);

			bytehashAdd(">string", STRING);
			bytehashAdd(">double", DOUBLE);
			bytehashAdd(">float", FLOAT);
			bytehashAdd(">long", LONG);
			bytehashAdd(">int", INTEGER);
			bytehashAdd(">integer", INTEGER);
			bytehashAdd(">short", SHORT);
			bytehashAdd(">byte", BYTE);
			bytehashAdd(">char", CHARACTER);
			bytehashAdd(">character", CHARACTER);
			bytehashAdd(">boolean", BOOLEAN);
			}

		Byte rv = (Byte)(bytehash.get(s.toLowerCase()));
		if (rv == null)
			{
			return INVALID;
			}

		return rv.byteValue();
		}


	static void print(String s)
		{
		System.out.print(s);
		}

	static void print(char c)
		{
		System.out.print(c);
		}

	static void println(String s)
		{
		System.out.println(s);
		}

	static void println()
		{
		println("");
		}

	void init()
		{
 		dictionary = new FijiDictionary();

		bytes = new Vector();
		objects = new Vector();

		if (globalDictionary == null)
			{
			// set our global dictionary to be this dictionary,
			globalDictionary = dictionary;

			boolean restoredValue = debugPrintTrace;
			debugPrintTrace = false; // false; // true;

			// add all this stuff,
			interpret(""
				+ " : >pfa >body ; inline "
				+ " : :noname \"\" create' postpone ] most-recent-word ; immediate "

				+ " : true 1 ; inline "
				+ " : false 0 ; inline "

				+ " : constant create , does> @ ; immediate "
				+ " : variable create 0 , ; immediate "
				+ " : cells ; immediate " // no runtime cost!
				+ " : 0?  0 =  ; inline "
				+ " : 0=  0 =  ; inline "
				+ " : 1+  1 +  ; inline "
				+ " : 1-  1 -  ; inline "
				+ " : +!  dup @ 1 + !  ; inline "
				+ " : over  1 pick  ; inline "
				+ " : tuck  dup rot rot  ; inline "
				+ " : 2dup  over over  ; inline "
				+ " : 2drop  drop drop  ; inline "
				+ " : doublequote  34C  ; inline "
				+ " : sp  32C  ; inline "

				// class constants
				+ " \"java.lang.String\" constant java.lang.String "
				+ " \"java.lang.Double\" constant java.lang.Double "
				+ " \"java.lang.Float\" constant java.lang.Float "
				+ " \"java.lang.Long\" constant java.lang.Long "
				+ " \"java.lang.Integer\" constant java.lang.Integer "
				+ " \"java.lang.Character\" constant java.lang.Character "
				+ " \"java.lang.Short\" constant java.lang.Short "
				+ " \"java.lang.Byte\" constant java.lang.Byte "
				+ " \"java.lang.Void\" constant java.lang.Void "

				// DPANS94 says you only need these six words:
				// 		 IF, THEN, BEGIN, AGAIN, UNTIL, and AHEAD
				// to define the rest of Forth's branching/looping
				// constructs.  These six, in turn, can be defined
				// using jump, branch0, jump-{forward,backward}-{push,fixup}
				
				// if/else/then
				+ " : if  postpone (branch0-check) (jump-forward-push) ; immediate "
				+ " : then  (jump-forward-fixup) ; immediate "
				+ " : endif  postpone then ; immediate "

				// begin/again/until
				+ " : begin (jump-backward-push) ; immediate "
				+ " : again  postpone (jump-check) (jump-backward-fixup) ; immediate "
				+ " : until  postpone (branch0-check) (jump-backward-fixup) ; immediate "

				// ahead
				+ " : ahead postpone (jump-check) (jump-forward-push) ; immediate "

				// here, then, are the DPANS94 definitions for the rest:
				+ " : while postpone if 1 cs-roll ; immediate "
				+ " : repeat postpone again postpone then ; immediate "
				+ " : else  postpone ahead 1 cs-roll postpone then ; immediate "

				// case/of/endof/endcase
				+ " : case 0 ; immediate "
				+ " : of 1 + postpone over postpone = postpone if postpone drop ; immediate "
				+ " : endof postpone else ; immediate "
				+ " : endcase postpone drop 0? do postpone then loop ; immediate "

				+ " : dumpword postpone see ; immediate "

				+ " : quit .\"It's not 'quit', it's 'bye'.  Remember that next time, huh?\n\" bye ; "

				// value / to  (will change when I add locals)
				+ " : value postpone constant ; immediate "
				+ " : to  "
				+ "		' "
				+ "		state? "
				+ "		if "
				+ "			dup "
				+ "			postpone literal postpone >body  "
				+ "			>cfa >serialnumber \"(do-local)\" '' >cfa >serialnumber = "
				+ "			if "
				+ "				postpone @ postpone ls-replace "
				+ "			else "
				+ "				postpone ! "
				+ "			endif "
				+ "		else "
				+ "			>body ! "
				+ "		then "
				+ "		; immediate "

				+ " : [else]  ( -- ) "
				+ "     1 begin                               "		// level
				+ "       begin "
				+ "         nextnonwhitetoken dup             "
	//			+ "         dup .\"^^elsegot->\" .. .\"<-^^ \" cr "
				+ "                               >length     "		// level token length-of-token
				+ "       while "									// level token
				+ "         dup \"[if]\" stricmp 0= "
				+ "         if                                "		// level token
				+ "           drop 1+                         "		// level'
				+ "         else                              "		// level token
				+ "           dup  \"[else]\" stricmp 0=      "		// level token token==else
				+ "           if                              "		// level token
				+ "              drop 1- dup if 1+ then       "		// level'
				+ "           else                            "		// level token
				+ "             dup \"[endif]\" stricmp 0=    "		// level token token==endif
				+ "             swap \"[then]\" stricmp 0=    "		// level token==endif token==then
	//			+ " 2dup .\"::> \" . . .\"<::\" "
				+ "             || if                         "		// level
				+ "               1-                          "		// level'
				+ "             then "
				+ "           then "
				+ "         then ?dup 0=  if exit then        "		// level'
				+ "       repeat  drop                        "		// level
				+ "     moretokens 0= until                   "		// level
				+ "     drop "
				+ " ;  immediate "

				+ " : [if]  ( flag -- ) "
				+ " 0= if postpone [else] then ;  immediate "

				+ " : [then]   ( -- )  ;  immediate "
				+ " : [endif]  ( -- )  ;  immediate "

				+ " : .s  ( -- ) "
				+ " .\"[| .s begin ] depth \" depth .. cr "
				+ "  depth 0 > if "
				+ "      1 depth 1- do "
				+ "          .\" | \" i 4 .r .\": \" "
				+ "          i 1- pick "
				+ "          dup "
				+ "		     .\"class \" >class . .\" \" "
				+ "          .\"== \" . "
				+ "          cr "
				+ "          -1 "
				+ "      +loop "
				+ "  endif "
				+ " .\"[| .s end ]\" cr "
				+ " ;"

				// ladies and gentlemen: may I present to you
				// the awe-inspiring VECTORED EXECUTION!
				// also known as function pointers in C.
				// without further ado: doer-make.
				+ " : nothing ; "

				+ " : doer postpone : postpone ; ; "

				+ " variable (make-marker) "

				+ " : (make-reset-marker) (make-marker) (make-marker) ! ; "
				+ " (make-reset-marker) "

				+ " : (make) "
				+ " 	r> dup 1+ 1 c+ dup 1+ 1 c+ swap @ body! "
				+ " 	dup @ ?dup "
				+ " 	if "
				+ " 		dup			" // r> marker marker
				+ " 		>bytex		" // r> marker marker.bytex "
				+ " 		rot			" // marker marker.bytex r> "
				+ " 		swap		" // marker r> marker.bytex "
				+ " 		bytex!		" // marker r>' "

				+ " 		swap		" // r>' marker "

				+ " 		>objectx	" // r>' marker.objectx "
				+ " 		objectx!	" // r>'' "

				+ " 		>r "
				+ " 	else "
				+ " 		drop "
				+ " 	then "
				+ " 	; "

				+ " : make "
				+ " 	state? "
				+ " 	if "
							// compile-time make
				+ " 		postpone (make) "

							// the following code (up to the "endif")
							// adds support for ";and" after chained "make" s
							// (like having an ";and" after the "recital" example
							// in Thinking Forth):

							// are marker and here from the same word?
				+ " 		(make-marker) @ >serialnumber "
				+ " 		here >serialnumber "
				+ " 		!= "

							// if they aren't, store a new marker
				+ " 		if "
				+ " 			here (make-marker) ! "
				+ " 		endif "

				+ " 		postpone push 0 , "

				+ " 	else "
							// runtime make
							// uses marker to store the tick of the word
				+ " 		' (make-marker) ! "
				+ " 		\"\" create' postpone ] "
				+ " 	then "
				+ " 	; immediate "

				+ " : ;and "
				+ " 	postpone exit here (make-marker) @ ! "
				+ " 	; immediate "

					// used to end a runtime make, *sigh*
				+ " : ;make "
				+ " 	postpone ; "
				+ " 	most-recent-word >body (make-marker) @ body! "
				+ " 	(make-reset-marker) "
				+ " 	; immediate "

				+ " : (do-local) @ ls-pick ; "

				// ANS suggested locals syntax!  feh.
				+ " : locals|  " // ( "name...name |" -- )
				+ " 	begin "
				+ " 	nextnonwhitetoken dup \"|\" != "
				+ " 	while "
				+ " 	1 (local) "
				+ " 	repeat "
				+ "		drop "
				+ " 	; immediate "

				+ " : local " // ( "name" -- )
				+ " 	nextnonwhitetoken dup >length (local) "
				+ " 	;  immediate "

				+ " : end-locals " // ( -- )
				+ " 	0 0 (local) ;  immediate "

				// the far-superior JH-locals syntax!  yay!
				+ " 1 value {no-comment-yet} "

				+ " : { "
				+ " 	0 >ls "
				+ " 	1 to {no-comment-yet} "
						// first, push all the strings onto the LS stack (to FILO them)
				+ " 	begin "
				+ " 		nextnonwhitetoken "
				+ " 		dup "
				+ " 		\"}\" != "
				+ " 	while "
				+ " 		dup "
				+ " 		\"--\" = "
				+ " 		if "
				+ " 			drop "
				+ " 			0 to {no-comment-yet} "
				+ " 		else "
				+ " 			{no-comment-yet} "
				+ " 			if "
				+ " 				ls> "
				+ " 				swap "
				+ " 				>ls "
				+ " 				1+ "
				+ " 				>ls "
				+ " 			else "
				+ " 				drop "
				+ " 			endif "
				+ " 		endif "
				+ " 	repeat "
				+ " 	drop "
				
						// now peel off the argument names from the LS stack and (local) them
				+ " 	ls> 0 " 
				+ " 	do "
				+ " 		ls> 1 (local) "
				+ " 	loop "
				+ " 	; immediate "

				// just in case
				+ " : } ; immediate "

				// FOR .. NEXT
				+ " : for postpone 0 postpone swap postpone ?do ; immediate "
				+ " : next postpone -1 postpone +loop ; immediate "


				+ " : executor nextnonwhitetoken dup create' finalize '' cfa! ; immediate "


   				// + ""
				);

			// and now give this interpreter its own fresh dictionary
			dictionary = new FijiDictionary();

			// more setup required:

			// assemble doLocal from spare parts
			FijiWord w = globalDictionary.lookup("(do-local)");
			doLocal = doDoes.copy();
			doLocal.bytes = w.PFA.bytes;
			doLocal.objects = w.PFA.objects;
			doLocal.serialNumber = w.CFA.serialNumber;

			debugPrintTrace = restoredValue;
			}
		}

	public FijiInterpreter()
		{
		init();
		}

	Object popReturn()
		{
		Object o = RS.lastElement();
		RS.removeElementAt(RS.size() - 1);
		return o;
		}

	void pushReturn(Object o)
		{
		RS.addElement(o);
		}

	Object pickReturn(int i)
		{
		return RS.elementAt((RS.size() - 1) - i);
		}

	void replaceReturn(int i, Object o)
		{
		RS.setElementAt(o, (RS.size() - 1) - i);
		}

	Object popControl()
		{
		Object o = CS.lastElement();
		CS.removeElementAt(CS.size() - 1);
		return o;
		}

	void pushControl(Object o)
		{
		CS.addElement(o);
		}

	Object pickControl(int i)
		{
		return CS.elementAt((CS.size() - 1) - i);
		}

	void replaceControl(int i, Object o)
		{
		CS.setElementAt(o, (CS.size() - 1) - i);
		}

	Object popLocal()
		{
		Object o = LS.lastElement();
		LS.removeElementAt(LS.size() - 1);
		return o;
		}

	void pushLocal(Object o)
		{
		LS.addElement(o);
		}

	Object pickLocal(int i)
		{
		return LS.elementAt((LS.size() - 1) - i);
		}

	void replaceLocal(int i, Object o)
		{
		LS.setElementAt(o, (LS.size() - 1) - i);
		}

	// top of stack
	Object tos()
		{
		return DS.lastElement();
		}

	Object pick(int i)
		{
		return DS.elementAt((DS.size() - 1) - i);
		}

	Object pop()
		{
		Object o = DS.lastElement();
		DS.removeElementAt(DS.size() - 1);
		return o;
		}

	void replace(int i, Object o)
		{
		DS.setElementAt(o, (DS.size() - 1) - i);
		}

	void push(Object o)
		{
		DS.addElement(o);
		}

	void push(double n)
		{
		push(new Double(n));
		}

	void push(float n)
		{
		push(new Float(n));
		}

	void push(long n)
		{
		push(new Long(n));
		}

	void push(int n)
		{
		push(new Integer(n));
		}

	void push(short n)
		{
		push(new Short(n));
		}

	void push(byte n)
		{
		push(new Byte(n));
		}

	void push(char n)
		{
		push(new Character(n));
		}

	void push(boolean n)
		{
		push(new Boolean(n));
		}

	FijiWord lookup(String s)
		{
		FijiWord w = null;
		if (newWord != null)
			{
			if (doesProgram != null && doesProgram.locals != null
				&& ((w = doesProgram.locals.lookup(s)) != null))
				return w;

			if (newWord.PFA.locals != null
				&& (w = newWord.PFA.locals.lookup(s)) != null)
				return w;
			}
		if ((w = dictionary.lookup(s)) != null)
			return w;
		if ((w = globalDictionary.lookup(s)) != null)
			return w;
		return null;
		}

	// when evaluating an expression between these two objects,
	// what should the resulting type be?
	static int DetermineType(Object o)
		{
		if (o instanceof String)
			return STRING;
		if (o instanceof Double)
			return DOUBLE;
		if (o instanceof Float)
			return FLOAT;
		if (o instanceof Long)
			return LONG;
		if (o instanceof Integer)
			return INTEGER;
		if (o instanceof Short)
			return SHORT;
		if (o instanceof Byte)
			return BYTE;
		if (o instanceof Character)
			return CHARACTER;
		if (o instanceof Boolean)
			return BOOLEAN;
		if (o instanceof FijiPointer)
			return POINTER;
		return INVALID;
		}

	static int DetermineTypePromotion(Object a, Object b)
		{
		int aType = DetermineType(a);
		int bType = DetermineType(b);

		return (aType < bType) ? aType : bType;
		}


	Object MathString(byte operand, String left, String right)
		{
		if (operand == ADD)
			return left + right;
	
		int comparison = left.compareTo(right);
		boolean rv;
						
		switch (operand)
			{
			case LESS_THAN:
				rv = (comparison < 0);
				break;
			case LESS_THAN_EQUALS:
				rv = (comparison <= 0);
				break;
			case EQUALS:
				rv = (comparison == 0);
				break;
			case GREATER_THAN_EQUALS:
				rv = (comparison >= 0);
				break;
			case GREATER_THAN:
				rv = (comparison > 0);
				break;
			case NOT_EQUAL_TO:
				rv = (comparison != 0);
				break;
			default:
				rv = false; // what the hell?
			}
		return new Integer(rv ? 1 : 0);
		}

	
	Object MathPointer(byte operand, FijiPointer left, Object right)
		{
		FijiPointer rv = left;
		switch (operand)
			{
			case ADD:
				rv.objectX += ((Integer)right).intValue();
				break;
			case SUBTRACT:
				rv.objectX -= ((Integer)right).intValue();
				break;
			case EQUALS:
				return new Integer((left.serialNumber == ((FijiPointer)right).serialNumber) ? 1 : 0);
			case NOT_EQUAL_TO:
				return new Integer((left.serialNumber != ((FijiPointer)right).serialNumber) ? 1 : 0);
			}

		return rv;
		}

	
	// I actually promote each type to the next higher
	// type on every add, multiply, or divide operation,
	// so I don't overflow.
	//   (byte)127 + 1 = (int)128
	//   (byte)127 * 2 = (int)254
	//   (byte)-128 - 1 = (int)-129
	// (I figure I only need to promote division for float)

	Object MathDouble(byte operand, double left, double right)
		{
		double rv = 0;
		switch (operand)
			{
			case NOT:
				rv = (left != 0) ? 0 : 1;
				break;
			case ADD:
				rv = left + right;
				break;
			case SUBTRACT:
				rv = left - right;
				break;
			case MULTIPLY:
				rv = left * right;
				break;
			case DIVIDE:
				rv = left / right;
				break;
			case LESS_THAN:
				rv = (left < right) ? 1.0D : 0.0D;
				break;
			case LESS_THAN_EQUALS:
				rv = (left <= right) ? 1.0D : 0.0D;
				break;
			case EQUALS:
				rv = (left == right) ? 1.0D : 0.0D;
				break;
			case GREATER_THAN_EQUALS:
				rv = (left >= right) ? 1.0D : 0.0D;
				break;
			case GREATER_THAN:
				rv = (left > right) ? 1.0D : 0.0D;
				break;
			case NOT_EQUAL_TO:
				rv = (left != right) ? 1.0D : 0.0D;
				break;
			}

		return new Double(rv);
		}

	
	Object MathFloat(byte operand, float left, float right)
		{
		float rv = 0;
		switch (operand)
			{
			case NOT:
				rv = (left != 0) ? 0 : 1;
				break;
			case ADD:
				rv = left + right;
				break;
			case SUBTRACT:
				rv = left - right;
				break;
			case MULTIPLY:
				rv = left * right;
				break;
			case DIVIDE:
				rv = left / right;
				break;
			case LESS_THAN:
				rv = (left < right) ? 1.0F : 0.0F;
				break;
			case LESS_THAN_EQUALS:
				rv = (left <= right) ? 1.0F : 0.0F;
				break;
			case EQUALS:
				rv = (left == right) ? 1.0F : 0.0F;
				break;
			case GREATER_THAN_EQUALS:
				rv = (left >= right) ? 1.0F : 0.0F;
				break;
			case GREATER_THAN:
				rv = (left > right) ? 1.0F : 0.0F;
				break;
			case NOT_EQUAL_TO:
				rv = (left != right) ? 1.0F : 0.0F;
				break;
			}

		return new Float(rv);
		}

	
	Object MathLong(byte operand, long left, long right)
		{
		long rv = 0;
		switch (operand)
			{
			case NOT:
				rv = (left != 0) ? 0 : 1;
				break;
			case ADD:
				rv = left + right;
				break;
			case SUBTRACT:
				rv = left - right;
				break;
			case MULTIPLY:
				rv = left * right;
				break;
			case DIVIDE:
				rv = left / right;
				break;
			case SHIFT_LEFT:
				rv = left << right;
				break;
			case SHIFT_RIGHT:
				rv = left >> right;
				break;
			case SHIFT_RIGHT0:
				rv = left >>> right;
				break;
			case BINARY_AND:
				rv = left & right;
				break;
			case BINARY_OR:
				rv = left | right;
				break;
			case BINARY_XOR:
				rv = left ^ right;
				break;
			case BOOLEAN_AND:
				rv = ((left != 0) && (right != 0)) ? 1 : 0;
				break;
			case BOOLEAN_OR:
				rv = ((left != 0) || (right != 0)) ? 1 : 0;
				break;
			case BOOLEAN_XOR:
				rv = ((left != 0) ? 1 : 0) ^ ((right != 0) ? 1 : 0);
				break;
			case LESS_THAN:
				rv = (left < right) ? 1 : 0;
				break;
			case LESS_THAN_EQUALS:
				rv = (left <= right) ? 1 : 0;
				break;
			case EQUALS:
				rv = (left == right) ? 1 : 0;
				break;
			case GREATER_THAN_EQUALS:
				rv = (left >= right) ? 1 : 0;
				break;
			case GREATER_THAN:
				rv = (left > right) ? 1 : 0;
				break;
			case NOT_EQUAL_TO:
				rv = (left != right) ? 1 : 0;
				break;
			}

		return new Long(rv);
		}

	
	Object MathInteger(byte operand, int left, int right)
		{
		int rv = 0;
		switch (operand)
			{
			case NOT:
				rv = (left != 0) ? 0 : 1;
				break;
			case ADD:
				rv = left + right;
				break;
			case SUBTRACT:
				rv = left - right;
				break;
			case MULTIPLY:
				rv = left * right;
				break;
			case DIVIDE:
				rv = left / right;
				break;
			case SHIFT_LEFT:
				rv = left << right;
				break;
			case SHIFT_RIGHT:
				rv = left >> right;
				break;
			case SHIFT_RIGHT0:
				rv = left >>> right;
				break;
			case BINARY_AND:
				rv = left & right;
				break;
			case BINARY_OR:
				rv = left | right;
				break;
			case BINARY_XOR:
				rv = left ^ right;
				break;
			case BOOLEAN_AND:
				rv = ((left != 0) && (right != 0)) ? 1 : 0;
				break;
			case BOOLEAN_OR:
				rv = ((left != 0) || (right != 0)) ? 1 : 0;
				break;
			case BOOLEAN_XOR:
				rv = ((left != 0) ? 1 : 0) ^ ((right != 0) ? 1 : 0);
				break;
			case LESS_THAN:
				rv = (left < right) ? 1 : 0;
				break;
			case LESS_THAN_EQUALS:
				rv = (left <= right) ? 1 : 0;
				break;
			case EQUALS:
				rv = (left == right) ? 1 : 0;
				break;
			case GREATER_THAN_EQUALS:
				rv = (left >= right) ? 1 : 0;
				break;
			case GREATER_THAN:
				rv = (left > right) ? 1 : 0;
				break;
			case NOT_EQUAL_TO:
				rv = (left != right) ? 1 : 0;
				break;
			}

		return new Integer(rv);
		}

	
	Object MathShort(byte operand, short left, short right)
		{
		int rv = 0;
		switch (operand)
			{
			case NOT:
				rv = (left != 0) ? 0 : 1;
				break;
			case ADD:
				rv = left + right;
				break;
			case SUBTRACT:
				rv = left - right;
				break;
			case MULTIPLY:
				rv = left * right;
				break;
			case DIVIDE:
				rv = left / right;
				break;
			case SHIFT_LEFT:
				rv = left << right;
				break;
			case SHIFT_RIGHT:
				rv = left >> right;
				break;
			case SHIFT_RIGHT0:
				rv = left >>> right;
				break;
			case BINARY_AND:
				rv = left & right;
				break;
			case BINARY_OR:
				rv = left | right;
				break;
			case BINARY_XOR:
				rv = left ^ right;
				break;
			case BOOLEAN_AND:
				rv = ((left != 0) && (right != 0)) ? 1 : 0;
				break;
			case BOOLEAN_OR:
				rv = ((left != 0) || (right != 0)) ? 1 : 0;
				break;
			case BOOLEAN_XOR:
				rv = ((left != 0) ? 1 : 0) ^ ((right != 0) ? 1 : 0);
				break;
			case LESS_THAN:
				rv = (left < right) ? 1 : 0;
				break;
			case LESS_THAN_EQUALS:
				rv = (left <= right) ? 1 : 0;
				break;
			case EQUALS:
				rv = (left == right) ? 1 : 0;
				break;
			case GREATER_THAN_EQUALS:
				rv = (left >= right) ? 1 : 0;
				break;
			case GREATER_THAN:
				rv = (left > right) ? 1 : 0;
				break;
			case NOT_EQUAL_TO:
				rv = (left != right) ? 1 : 0;
				break;
			}

		return new Integer(rv);
		}

	
	Object MathByte(byte operand, byte left, byte right)
		{
		int rv = 0;
		switch (operand)
			{
			case NOT:
				rv = (left != 0) ? 0 : 1;
				break;
			case ADD:
				rv = left + right;
				break;
			case SUBTRACT:
				rv = left - right;
				break;
			case MULTIPLY:
				rv = left * right;
				break;
			case DIVIDE:
				rv = left / right;
				break;
			case SHIFT_LEFT:
				rv = left << right;
				break;
			case SHIFT_RIGHT:
				rv = left >> right;
				break;
			case SHIFT_RIGHT0:
				rv = left >>> right;
				break;
			case BINARY_AND:
				rv = left & right;
				break;
			case BINARY_OR:
				rv = left | right;
				break;
			case BINARY_XOR:
				rv = left ^ right;
				break;
			case BOOLEAN_AND:
				rv = ((left != 0) && (right != 0)) ? 1 : 0;
				break;
			case BOOLEAN_OR:
				rv = ((left != 0) || (right != 0)) ? 1 : 0;
				break;
			case BOOLEAN_XOR:
				rv = ((left != 0) ? 1 : 0) ^ ((right != 0) ? 1 : 0);
				break;
			case LESS_THAN:
				rv = (left < right) ? 1 : 0;
				break;
			case LESS_THAN_EQUALS:
				rv = (left <= right) ? 1 : 0;
				break;
			case EQUALS:
				rv = (left == right) ? 1 : 0;
				break;
			case GREATER_THAN_EQUALS:
				rv = (left >= right) ? 1 : 0;
				break;
			case GREATER_THAN:
				rv = (left > right) ? 1 : 0;
				break;
			case NOT_EQUAL_TO:
				rv = (left != right) ? 1 : 0;
				break;
			}

		return new Short((short)rv);
		}

	Object MathCharacter(byte operand, char left, char right)
		{
		int rv = 0;
		switch (operand)
			{
			case ADD:
				return new Character((char)(left + right));
			case SUBTRACT:
				return new Character((char)(left - right));
			case MULTIPLY:
			case DIVIDE:
			case SHIFT_LEFT:
			case SHIFT_RIGHT:
			case SHIFT_RIGHT0:
			case BINARY_AND:
			case BINARY_OR:
			case BINARY_XOR:
			case BOOLEAN_AND:
			case BOOLEAN_OR:
			case BOOLEAN_XOR:
				// NEEDSWORK: throw an exception
				rv = 0;
				break;
			case LESS_THAN:
				rv = (left < right) ? 1 : 0;
				break;
			case LESS_THAN_EQUALS:
				rv = (left <= right) ? 1 : 0;
				break;
			case EQUALS:
				rv = (left == right) ? 1 : 0;
				break;
			case GREATER_THAN_EQUALS:
				rv = (left >= right) ? 1 : 0;
				break;
			case GREATER_THAN:
				rv = (left > right) ? 1 : 0;
				break;
			case NOT_EQUAL_TO:
				rv = (left != right) ? 1 : 0;
				break;
			}

		return new Integer((short)rv);
		}

	static boolean debugPrintTrace = false; // true;


	// fix up jump-to-jumps
	void optimizeJump(int byteX, int objectX)
		{
		int jumpToBytes = ((Number)program.objects[objectX]).intValue();
		int jumpToObjects = ((Number)program.objects[objectX + 1]).intValue();

		switch (program.bytes[jumpToBytes])
			{
			case JUMP_CHECK:
				optimizeJump(jumpToBytes, jumpToObjects);
				// intentional fall-through

			case JUMP:
				program.objects[objectX] = program.objects[jumpToObjects];
				program.objects[objectX + 1] = program.objects[jumpToObjects + 1];
				// intentional fall-through

			default:
				break;
			}

		// only do this the first time--
		// the check opcode is always +1 the non-check opcode
		program.bytes[byteX]--;
		}



	void finalizeEntry()
		{
		if (creating)
			{
			creating = false;
			compiling = false;
			bytes.addElement(new Byte(EXIT));

			byte[] pfaBytes = new byte[bytes.size()];
			for (int i = 0; i < bytes.size(); i++)
				{
				pfaBytes[i] = ((Byte)(bytes.elementAt(i))).byteValue();
				}
			bytes.removeAllElements();

			Object[] pfaObjects = new Object[objects.size()];
			objects.copyInto(pfaObjects);
			objects.removeAllElements();

			newWord.PFA.bytes = pfaBytes;
			newWord.PFA.objects = pfaObjects;
			if (doesProgram != null)
				{
				doesProgram.bytes = pfaBytes;
				doesProgram.objects = pfaObjects;
				doesProgram = null;
				}
			}
		}

	FijiPointer ensureFinal(FijiPointer p)
		{
		if (!creating
			|| (newWord.PFA.serialNumber != p.serialNumber))
			return p;
		
		finalizeEntry();
		return newWord.PFA;
		}


	void pushFixup(byte operand)
		{
		FijiCallFrame f = new FijiCallFrame(program.copy());

		f.byteX = bytes.size();
		f.objectX = objects.size();
		f.operand = operand;

		pushControl(f);

		if (operand >= JUMP_FORWARD_PUSH)
			{
			objects.addElement(null);
			objects.addElement(null);
			}
		}


	void jump()	
		{
		program.byteX = ((Number)program.objects[program.objectX++]).intValue();
		program.objectX = ((Number)program.objects[program.objectX]).intValue();
		}

	void skipJump()
		{
		program.objectX += 2;
		}


	void fixupForward(FijiCallFrame f)
		{
		objects.setElementAt(new Integer(bytes.size()), f.objectX);
		objects.setElementAt(new Integer(objects.size()), f.objectX + 1);
		}

	
	void fixupBackward(FijiCallFrame f)
		{
		objects.addElement(new Integer(f.byteX));
		objects.addElement(new Integer(f.objectX));
		}

	
	public void interpret(FijiPointer RunThisProgram)
		{
		if (!noByeYet)
			return;

		program = RunThisProgram;

		// jdb gets really confused when you have variables
		// with the same name in different nested scopes
		// in the same function.  so, I declare all the local
		// variables I use in the case statements right here.
		byte b;
		char c;
		FijiCallFrame f;
		int i, j, stopAt;
		Object o, o2;
		FijiPointer p, q;
		String s, t;
		FijiWord w;
		byte operand;
		boolean skip, swap;

		boolean KeepGoing = true;
		while (KeepGoing)
			{
			operand = program.bytes[program.byteX++];

			if (debugPrintTrace)
				{
				println("[+ " + "rs " + RS.toString() + " ]");
				println(" | " + "ds " + DS.toString());
				println(" | bytex, objectx (" + (program.byteX - 1) + ", " + program.objectX + ") "
					+ " == " + byteToString[operand + 128] + " (" + operand + "), ");
				println("[+]");
				}

			switch (operand)
				{
				case -1:
				case 0:	
				case 1:	
				case 2:	
				case 3:	
				case 4:	
				case 5:	
				case 6:	
				case 7:	
				case 8:	
				case 9:	
				case 10:
				case 11:
				case 12:
				case 13:
				case 14:
				case 15:
				case 16:
					push((int)operand);
					break;

				case NOT:
					switch (DetermineType(tos()))
						{
						case DOUBLE:
							replace(0, MathDouble(operand, ((Number)tos()).doubleValue(), (double)0));
							break;
						case FLOAT:
							replace(0, MathFloat(operand, ((Number)tos()).floatValue(), (float)0));
							break;
						case LONG:
							replace(0, MathLong(operand, ((Number)tos()).longValue(), (long)0));
							break;
						case INTEGER:
							replace(0, MathInteger(operand, ((Number)tos()).intValue(), (int)0));
							break;
						case SHORT:
							replace(0, MathShort(operand, ((Number)tos()).shortValue(), (short)0));
							break;
						case BYTE:
							replace(0, MathByte(operand, ((Number)tos()).byteValue(), (byte)0));
							break;
						default:
							println("unary operation not supported for top two stack elements");
						}
					break;

				case ADD:
				case SUBTRACT:
				case MULTIPLY:
				case DIVIDE:
				case SHIFT_LEFT:
				case SHIFT_RIGHT:
				case SHIFT_RIGHT0:
				case BINARY_AND:
				case BINARY_OR:
				case BINARY_XOR:
				case BOOLEAN_AND:
				case BOOLEAN_OR:
				case BOOLEAN_XOR:
				case LESS_THAN:
				case LESS_THAN_EQUALS:
				case EQUALS:
				case GREATER_THAN:
				case GREATER_THAN_EQUALS:
				case NOT_EQUAL_TO:
					{
					Object Secondary = pop();
					Object Primary = pop();
					int ResultingType;

					ResultingType = DetermineTypePromotion(Primary, Secondary);

					swap = false;
					switch (ResultingType)
						{
						case POINTER:
							// code assumes that primary is the FijiPointer
							swap = (Secondary instanceof FijiPointer);
							break;
						case STRING:
							// maintain the order of the strings
							swap = true;
							break;
						}

					if (swap)
						{
						Object tmp = Primary;
						Primary = Secondary;
						Secondary = tmp;
						}

					switch (ResultingType)
						{
						case STRING:
							push(MathString(operand,
								Primary.toString(),
								Secondary.toString()));
							break;
						case POINTER:
							push(MathPointer(operand,
								(FijiPointer)Primary,
								Secondary));
							break;
						case DOUBLE:
							push(MathDouble(operand,
								((Number)Primary).doubleValue(),
								((Number)Secondary).doubleValue()));
							break;
						case FLOAT:
							push(MathFloat(operand,
								((Number)Primary).floatValue(),
								((Number)Secondary).floatValue()));
							break;
						case LONG:
							push(MathLong(operand,
								((Number)Primary).longValue(),
								((Number)Secondary).longValue()));
							break;
						case INTEGER:
							push(MathInteger(operand,
								((Number)Primary).intValue(),
								((Number)Secondary).intValue()));
							break;
						case SHORT:
							push(MathShort(operand,
								((Number)Primary).shortValue(),
								((Number)Secondary).shortValue()));
							break;
						case CHARACTER:
							switch (DetermineType(Secondary))
								{
								case CHARACTER:
									c = ((Character)Secondary).charValue();
									break;
								default:
									c = (char)(((Number)Secondary).intValue());
									break;
								}
							push(MathCharacter(operand,
								((Character)Primary).charValue(),
								c));
							break;
						case BYTE:
							push(MathByte(operand,
								((Number)Primary).byteValue(),
								((Number)Secondary).byteValue()));
							break;
						default:
							println("math operand not supported for top two stack elements");
						}
					}
					break;


				// jumping and branching
				case BRANCH0_CHECK:
				case JUMP_CHECK:
					optimizeJump(program.byteX - 1, program.objectX);

					// intentional fall-through

				case BRANCH0:
					if (operand != JUMP_CHECK)
						{
						o = pop();

						if (debugPrintTrace)  println("<<branch0 \"" + o.toString() + "\">> ");

						if (o == null)
							skip = false;
						else
							{
							if (o instanceof FijiPointer)
								skip = true;
							else if (o instanceof Boolean)
								skip = ((Boolean)o).booleanValue();
							else
								skip = (((Number)o).intValue() != 0);
							}

						if (skip)
							{
							skipJump();
							break;
							}
						}

					// intentional fall-through

				case JUMP:
					jump();
					break;

				case NULL:
					push(null);
					break;

				case HERE:
					if (!creating)
						println("calling here while not creating!  ignoring.");
					else
						{
						p = newWord.PFA.copy();
						p.objectX = objects.size();
						p.byteX = bytes.size();
						push(p);
						}
					break;

				// NEEDSWORK:
				// change do/loop so that the three pops are done _after_ exiting the loop
				// that way leave becomes a simple jump

				case I_QUESTION_DO_CHECK:
					optimizeJump(program.byteX - 1, program.objectX);
					// intentional fall-through

				case I_QUESTION_DO:
					if ((((Number)tos()).intValue()) != (((Number)pick(1)).intValue()))
						{
						skipJump();
						break;
						}
					pop();
					pop();
					jump();
					break;

				case I_PREP_DO:
					i = ((Number)pop()).intValue();
					stopAt = ((Number)pop()).intValue();
					pushReturn(new Integer(stopAt));
					pushReturn(new Integer(i));
					pushReturn(new Integer((i < stopAt) ? 1 : 0));
					break;

				case I_DO_CHECK:
					optimizeJump(program.byteX - 1, program.objectX);
					// intentional fall-through

				case I_DO:
					// in this case statement, j will play the
					// part of (original-i < stopAt).
					stopAt = ((Number)pickReturn(2)).intValue();
					i =      ((Number)pickReturn(1)).intValue();
					j =      ((Number)pickReturn(0)).intValue();
					if ((((i < stopAt) ? 1 : 0) ^ j) != 0)
						{
						popReturn();
						popReturn();
						popReturn();
						jump();
						}
					else
						skipJump();
					break;

				case I:
				case J:
				case K:
					push(pickReturn((3 * (operand - I)) + 1));
					break;

				case I_LEAVE_CHECK:
					optimizeJump(program.byteX - 1, program.objectX);
					// intentional fall-through

				case I_LEAVE:
					popReturn();
					popReturn();
					popReturn();
					jump();
					break;

				case I_LOOP_CHECK:
				case I_PLUS_LOOP_CHECK:
					optimizeJump(program.byteX - 1, program.objectX);
					// intentional fall-through
					// but first, fool the if (operand == ) below
					operand--;

				case I_LOOP:
				case I_PLUS_LOOP:
					if (operand == I_LOOP)
						i = 1;
					else
						i = ((Number)pop()).intValue();

					i += ((Number)pickReturn(1)).intValue();
					replaceReturn(1, new Integer(i));
					jump();

					break;

				case QUESTION_DO:
					handleLater(I_QUESTION_DO_CHECK, null, null);
					pushFixup(QUESTION_DO);

					// intentional fall-through

				case DO:
					handleLater(I_PREP_DO, null, null);
					pushFixup(DO);
					handleLater(I_DO_CHECK, null, null);
					if (operand == QUESTION_DO)
						{
						// swap QUESTION_DO above DO on
						// control stack, for LOOP's sakes
						o = popControl();
						o2 = popControl();
						pushControl(o);
						pushControl(o2);
						}
					break;

				case LOOP:
				case PLUS_LOOP:
					{
					// NEEDSWORK:
					// all this DO stuff is pretty hacked up.

					// a mild hack:
					// first, push two garbage objects
					// this will allow the LEAVE fixupForward()
					// to fixup to the correct object count
					o = new Object();
					handleLater((byte)(operand + 2), o, o);
					for (;;)
						{
						f = (FijiCallFrame)popControl();
						switch (f.operand)
							{
							case LEAVE:
							case QUESTION_DO:
								fixupForward(f);
								// intentional fall-through
							default:
								continue;
							case DO:
								break;
							}
						// if we reach here, we found a DO
						break;
						}
					// now, remove those two garbage elements
					// because fixupBackward() will add the
					// correct backward fixups
					objects.removeElementAt(objects.size() - 1);
					objects.removeElementAt(objects.size() - 1);

					// now fix our (loop) or (+loop) jump backwards,
					fixupBackward(f);
					// and finally, fix our (do) jump forwards.
					fixupForward(f);
					}
					break;

				case LEAVE:
					bytes.addElement(new Byte(I_LEAVE_CHECK));
					// intentional fall-through

				// both JUMP_FORWARD_PUSH and JUMP_BACKWARD_FIXUP
				// assume you've already compiled the bytecode
				// for the jump or branch
				case JUMP_FORWARD_PUSH:
				case JUMP_BACKWARD_PUSH:
					pushFixup(operand);
					break;

				case JUMP_FORWARD_FIXUP:
				case JUMP_BACKWARD_FIXUP:
					// cycle the LEAVES off the control stack,
					i = 0;
					for (; ; )
						{
						f = (FijiCallFrame)popControl();
						if (f.operand != LEAVE)
							break;
						i++;
						push(f);
						}

					if (operand == JUMP_FORWARD_FIXUP)
						fixupForward(f);
					else
						fixupBackward(f);

					for (; i > 0; i--)
						{
						o = pop();
						pushControl(o);
						}
					break;

				// hard-coded strings
				case DOT_QUOTE:
				case QUOTE:
					// let's pretend t stands for Total
					t = "";
					while (moreTokens())
						{
						s = nextToken();
						int quoteIndex = s.indexOf('"');
						if (quoteIndex != -1)
							{
							t += s.substring(0, quoteIndex);
							break;
							}
						t += s;
						}

					handleObject(t);
					if (operand == DOT_QUOTE)
						handleByteCode(DOT_DOT);
	
					break;


				// stack manipulation
				case QUESTION_DUP:
					o = tos();
					if (o == null)
						break;
					if ((o instanceof Number) && (((Number)o).intValue() == 0))
						break;

					// intentional fall-through

				case DUP:
					o = tos();
					if (o instanceof FijiPointer)
						push(((FijiPointer)o).copy());
					else if (o instanceof FijiWord)
						push(((FijiWord)o).copy());
					else
						push(o);
					break;

				case TO_R:
					pushReturn(pop());
					break;

				case R_FROM:
					push(popReturn());
					break;

				case R_FETCH:
					push(pickReturn(0));
					break;

				case DROP:
					pop();
					break;

				case PUSH:
					push(program.objects[program.objectX++]);
					break;

				case REPLACE:
					i = ((Number)pop()).intValue();
					o = pop();
					replace(i, o);
					break;

				case LITERAL:
					bytes.addElement(new Byte(PUSH));
					objects.addElement(pop());
					break;

				case SWAP:
					i = DS.size() - 1;
					o = DS.elementAt(i);
					DS.setElementAt(DS.elementAt(i - 1), i);
					DS.setElementAt(o, i - 1);
					break;

				case ROT:
					i = DS.size() - 1;
					o = DS.elementAt(i - 2);
					DS.setElementAt(DS.elementAt(i - 1), i - 2);
					DS.setElementAt(DS.elementAt(i), i - 1);
					DS.setElementAt(o, i);
					break;

				case ROLL:
					stopAt = ((Number)pop()).intValue();

					o = pick(stopAt);
					for (i = stopAt; i > 0; i--)
						replace(i, pick(i - 1));
					replace(0, o);
					break;

				case CS_ROLL:
					stopAt = ((Number)pop()).intValue();

					o = pickControl(stopAt);
					for (i = stopAt; i > 0; i--)
						replaceControl(i, pickControl(i - 1));
					replaceControl(0, o);
					break;

				case PICK:
					i = ((Number)pop()).intValue();
					push(pick(i));
					break;

				case CS_PICK:
					i = ((Number)pop()).intValue();
					push(pickControl(i));
					break;

				case LS_PICK:
					i = ((Number)pop()).intValue();
					push(pickLocal(i));
					break;

				case LS_REPLACE:
					i = ((Number)pop()).intValue();
					replaceLocal(i, pop());
					break;

				case TO_LS:
					pushLocal(pop());
					break;

				case LS_FROM:
					push(popLocal());
					break;

				case STORE:
					p = (FijiPointer)pop();
					o = pop();
					if ((p.serialNumber == newWord.PFA.serialNumber) && (creating))
						objects.setElementAt(o, p.objectX);
					else
						p.objects[p.objectX] = o;
					break;

				case C_MINUS:
				case C_PLUS:
					i = ((Number)pop()).intValue();
					p = (FijiPointer)tos();
					if (operand == C_MINUS)
						i = 0 - i;
					p.byteX += i;
					break;

				case C_STORE:
					{
					p = (FijiPointer)pop();
					Byte byteObject = (Byte)pop();
					if ((p.serialNumber == newWord.PFA.serialNumber) && (creating))
						bytes.setElementAt(byteObject, p.byteX);
					else
						p.bytes[p.byteX] = byteObject.byteValue();
					}
					break;

				case DEPTH:
					push(DS.size());
					break;


				case C_FETCH:
					p = (FijiPointer)pop();
					if ((p.serialNumber == newWord.PFA.serialNumber) && (creating))
						push(bytes.elementAt(p.byteX));
					else
						push(p.bytes[p.byteX]);

				// QUESTION actually executes fetch and then falls through the case
				// to dot--it actually puts the value on the stack temporarily
				case QUESTION:
				case QUESTION_X:
				case QUESTION_O:
				case QUESTION_B:
				case QUESTION_QUESTION:
				case QUESTION_QUESTION_X:
				case QUESTION_QUESTION_O:
				case QUESTION_QUESTION_B:

				case FETCH:
					p = (FijiPointer)pop();
					if ((p.serialNumber == newWord.PFA.serialNumber) && (creating))
						push(objects.elementAt(p.objectX));
					else
						push(p.objects[p.objectX]);
					if (operand == FETCH)
						break;

					// change this to the equivalent DOT operand
					operand += (DOT - QUESTION);
					// intentional fall-through

				case DOT:
				case DOT_X:
				case DOT_O:
				case DOT_B:
				case DOT_DOT:
				case DOT_DOT_X:
				case DOT_DOT_O:
				case DOT_DOT_B:
					o = pop();

					if (false)
						print("<<" + o.getClass().toString() + ">> ");

					switch (operand)
						{
						case DOT:
						case DOT_DOT:
							print(o.toString());
							break;
						default:
							if (o instanceof Number)
								{
								Long l = null;
								Number n = null;
								if (o instanceof Long)
									l = (Long)o;
								else
									n = (Number)o;
								switch (operand)
									{
									case DOT_X:
									case DOT_DOT_X:
										if (l != null)
											print(Long.toHexString(l.longValue()));
										else
											print(Integer.toHexString(n.intValue()));
										break;
									case DOT_O:
									case DOT_DOT_O:
										if (l != null)
											print(Long.toOctalString(l.longValue()));
										else
											print(Integer.toOctalString(n.intValue()));
										break;
									case DOT_B:
									case DOT_DOT_B:
										if (l != null)
											print(Long.toBinaryString(l.longValue()));
										else
											print(Integer.toBinaryString(n.intValue()));
										break;
									}
								}
							else
								print(o.toString());
						}

					// print trailing space if appropriate
					switch (operand)
						{
						case DOT:
						case DOT_X:
						case DOT_O:
						case DOT_B:
							print(" ");
							break;
						}
					break;

				case DOT_R:
					i = ((Number)pop()).intValue();
					s = pop().toString();
					print(padString(s, i));
					break;

				case EMIT:
					c = (char)(((Number)pop()).intValue());
					print(c);
					break;

				case CR:
					println();
					break;

				case CALL:
				case EXECUTE:
					if (operand == CALL)
						w = (FijiWord)(program.objects[program.objectX++]);
					else
						w = (FijiWord)pop();
					if (debugPrintTrace) println("[[ EXECUTING \"" + w.name + "\" ]]");
					w.execute(this);
					break;

				case TO_BODY:
					w = (FijiWord)pop();
					push(w.PFA);
					break;

				case TO_CFA:
					w = (FijiWord)pop();
					push(w.CFA);
					break;

				case BODY_STORE:
					w = (FijiWord)pop();
					p = (FijiPointer)pop();
					w.PFA = p;
					break;

				case CFA_STORE:
					w = (FijiWord)pop();
					p = (FijiPointer)pop();
					w.CFA = p;
					break;

				case TO_BYTEX:
					p = (FijiPointer)pop();
					push(p.byteX);
					break;

				case TO_OBJECTX:
					p = (FijiPointer)pop();
					push(p.objectX);
					break;

				case TO_SERIALNUMBER:
					p = (FijiPointer)pop();
					push(p.serialNumber);
					break;

				case BYTEX_STORE:
					i = ((Number)pop()).intValue();
					p = (FijiPointer)tos();
					p.byteX = i;
					break;

				case OBJECTX_STORE:
					i = ((Number)pop()).intValue();
					p = (FijiPointer)tos();
					p.objectX = i;
					break;

				case BRACKET_TICK:
				case TICK:
				case TICK_PRIME:
					if (operand == TICK_PRIME)
						s = (String)pop();
					else
						s = nextNonWhiteToken();
					if (s != null)
						{
						w = lookup(s);
						if (operand == BRACKET_TICK)
							handleLater(PUSH, w, null);
						else
							push(w);
						}
					break;

				case POSTPONE:
					if (compiling)
						postponeNext = true;
					else
						println("calling postpone while not compiling!  ignoring.");
					break;

				case HANDLE_POSTPONED_WORD:
					w = (FijiWord)program.objects[program.objectX++];
					handleWord(w);
					break;

				case HANDLE_POSTPONED_BYTECODE:
					b = program.bytes[program.byteX++];
					handleByteCode(b);
					break;

				case COMPILE_BYTE:
					bytes.addElement(new Byte(program.bytes[program.byteX++]));
					break;

				case COMPILE_OBJECT:
					objects.addElement((program.objects[program.objectX++]));
					break;

				case LEFT_SQUARE:
					compiling = false;
					break;

				case RIGHT_SQUARE:
					compiling = true;
					break;

				case STATE_QUESTION:
					push(new Integer(compiling ? 1 : 0));
					break;

				case RECURSE:
					if (!compiling)
						println("calling recurse while not compiling!  ignoring.");
					else
						handleLater(CALL, newWord, null);
					break;

				case COLON:
				case COLON_PRIME:
					if (compiling)
						{
						println("executing colon while already compiling!  ignoring.");
						break;
						}

					// intentional fallthrough
				case CREATE:
				case CREATE_PRIME:
					if (!moreTokens())
						{
						println("executing \"create\" or \":\" as last word in stream!  ignoring.");
						}
					else
						{
						finalizeEntry();
						creating = true;
						String name;
						if ((operand == CREATE) || (operand == COLON))
							name = nextNonWhiteToken();
						else
							name = (String)pop();
						newWord = new FijiWord(name);
						newWord.PFA = new FijiPointer();
						dictionary.add(newWord);
						if ((operand == COLON) || (operand == COLON_PRIME))
							{
							compiling = true;
							newWord.CFA = doColon;
							newWord.smudged = true;
							}
						bytes.removeAllElements();
						objects.removeAllElements();
						if (debugPrintTrace)
							println("[[ CREATING: \""	+ name + "\" ]] ");
						}
					break;

				case DOES_GT:
					if (!compiling)
						println("calling does> when not compiling!  ignoring.");
					else
						{
						p = doDoes.copy();
						doesProgram = p;

						handleLater(DOES_SET_CFA, p, null);
						handleLater(EXIT, null, null);

						p.byteX = bytes.size();
						p.objectX = objects.size();
						}
					break;

				case IMMEDIATE:
					if (newWord != null)
						newWord.immediate = true;
					break;

				case INLINE:
					if (newWord != null)
						newWord.inline = true;
					break;

				case COMMA:
					objects.addElement(pop());
					break;

				case C_COMMA:
					bytes.addElement((Byte)pop());
					break;

				case C_ALLOT:
				case ALLOT:
					stopAt = ((Number)pop()).intValue();

					if (operand == ALLOT)
						o = new Integer(0);
					else
						o = new Byte((byte)0);

					for (i = 0; i < stopAt; i++)
						if (operand == ALLOT)
							objects.addElement(o);
						else
							bytes.addElement(o);
					break;

				case SEMICOLON:
					compiling = false;
					newWord.smudged = false;
					// intentional fall-through

				case DOES_SET_CFA:
					if (operand == DOES_SET_CFA)
						newWord.CFA = (FijiPointer)program.objects[program.objectX++];

					// intentional fall-through

				case FINALIZE:
					if (!creating)
						println("executing finalize (semicolon) without creating something first!  ignoring.");
					else
						finalizeEntry();
					break;


				// inspectors
				case TO_CLASS:
					o = pop();
					push(o.getClass().getName());
					break;

				case TO_LENGTH:
					o = pop();
					if (o instanceof String)
						push(((String)o).length());
					else
						{
						p = (FijiPointer)o;
						if (p.objects == null)
							push(-1);
						else
							push(p.objects.length);
						}
					break;

				case C_TO_LENGTH:
					p = (FijiPointer)pop();
					if (p.bytes == null)
						push(-1);
					else
						push(p.bytes.length);
					break;

				case NEXT_NONWHITE_TOKEN:
				case NEXT_TOKEN:
					s = (operand == NEXT_TOKEN)
						? nextToken() : nextNonWhiteToken();
					if (s == null)
						s = "";
					push(s);
					break;

				case MORE_TOKENS:
					push((int)(moreTokens() ? 1 : 0));
					break;

				case PUT_BACK_TOKEN:
					s = (String)pop();
					putBackToken(s);
					break;

				// casts
				case STRING:
					replace(0, tos().toString());
					break;

				case DOUBLE:
					replace(0, new Double(((Number)tos()).doubleValue()));
					break;

				case FLOAT:
					replace(0, new Float(((Number)tos()).floatValue()));
					break;

				case LONG:
					replace(0, new Long(((Number)tos()).longValue()));
					break;

				case INTEGER:
					replace(0, new Integer(((Number)tos()).intValue()));
					break;

				case SHORT:
					replace(0, new Short(((Number)tos()).shortValue()));
					break;

				case CHARACTER:
					replace(0, new Character( (char) (((Number)tos()).intValue()) ) );
					break;

				case BYTE:
					replace(0, new Byte(((Number)tos()).byteValue()));
					break;

				case BOOLEAN:
					o = tos();
					if (o instanceof Boolean)
						break;
					else if (o instanceof String)
						o = new Boolean(o.toString());
					else
						o = new Boolean((((Number)o).intValue()) != 0);
					replace(0, o);
					break;

				case BACKSLASH:
					while (moreTokens())
						{
						s = nextToken();
						if ((s.indexOf('\r') != -1)
							|| (s.indexOf('\n') != -1))
							break;
						}
					break;

				case LEFT_PAREN:
					while (moreTokens())
						{
						s = nextToken();
						// here, i is the index of the right paren
						i = s.indexOf(')');
						if (i != -1)
							{
							if (s.length() > (i + 1))
								putBackToken(s.substring(i + 1));
							break;
							}
						}
					break;

				case ADD_PREFIX:
					s = (String)pop();
					prefixVector.addElement(s);
					break;

				case REMOVE_PREFIX:
					s = (String)pop();
					prefixVector.removeElement(s);
					break;

				case STRICMP:
					t = (String)pop();
					s = (String)pop();
					i = s.toLowerCase().compareTo(t.toLowerCase());
					if (debugPrintTrace)
						println("### stricmp(\"" + s + "\", \"" + t + "\") == " + i + " ###");
					push(new Integer(i));
					break;

				case STRTOK:
					{
					t = (String)pop();
					s = (String)pop();
					
					i = 0;
					StringTokenizer tk = new StringTokenizer(s, t, true);
					while (tk.hasMoreTokens())
						{
						i++;
						push(tk.nextToken());
						}
					push(new Integer(i));
					}
					break;

				case SUBSTR:
					stopAt = ((Number)pop()).intValue();
					i = ((Number)pop()).intValue();
					s = (String)pop();
					push(s.substring(i, stopAt));
					break;

				case POPCHAR:
					s = (String)pop();
					if (s.length() > 0)
						{
						push(s.substring(1));
						push(new Character(s.charAt(0)));
						}
					else
						{
						push(s);
						push(new Character((char)0));
						}
					break;

				case EXIT:
					if (program.localCount != 0)
						{
						stopAt = ((Integer)popLocal()).intValue();
						for (i = 0; i < stopAt; i++)
							popLocal();
						}

					if (RS.size() == 0)
						KeepGoing = false;
					else
						program = (FijiPointer)popReturn();
					break;

				case BYE:
					noByeYet = false;
					// intentional fall-through

				case INTERNAL_STOP:
					KeepGoing = false;
					break;

				case INCLUDE:
					s = (String)pop();
					interpretFile(s);
					break;

				case EVALUATE:
					s = (String)pop();
					interpret(s);
					break;

				case CLOCK:
					push(new Long(System.currentTimeMillis()));
					break;

				case MOVE:
					stopAt = ((Number)pop()).intValue();
					q = (FijiPointer)pop();
					p = (FijiPointer)pop();
					for (i = 0; i < stopAt; i++)
						{
						q.objects[q.objectX + i] = p.objects[p.objectX + i];
						}
					break;

				case SEE:
					if (!moreTokens())
						{
						println("executing \"see\" as last word in stream!  ignoring.");
						}
					else
						{
						s = nextNonWhiteToken();
						w = lookup(s);
						print("[| see \"" + s + "\" begin ] " + w.toString());
						if (w == null)
							println("does not exist! ]");
						else
							{
							boolean isVariable = (w.CFA == FijiWord.doVariable);
							println("");
							print(" | CFA: ");
							if (isVariable)
								println("(variable)");
							else if (w.CFA == doColon)
								println("(do-colon) " + w.CFA.toString());
							else if (w.CFA instanceof FijiDoDoes)
								{
								println("(do-does)");
								println(" |   locals: "
									+ (w.CFA.locals == null ? "none" : w.CFA.locals.toString()));
								dumpWordProgram(w.CFA, true);
								}
							else if (w.CFA == doLocal)
								println("(do-local)");
							else if (w.CFA != null)
								println(w.CFA.toString());
							else
								println("null!");
							println(" | PFA: " + w.PFA.toString());
							println(" |   locals: "
								+ (w.PFA.locals == null ? "none" : w.PFA.locals.toString()));
							println(" |   byteX: " + w.PFA.byteX);
							println(" |   objectX: " + w.PFA.objectX);
							print(" |   Program: ");
							if (w.PFA.bytes == null) 
								println(" null!  must not be finalized yet.");
							else
								{
								println(w.PFA.bytes.length + " byte codes, "
									 + w.PFA.objects.length + " objects");
								dumpWordProgram(w.PFA, !isVariable);
								}
							println("[| see \"" + s + "\" end ]");
							}
						}
					break;
				
				case PUSH_FIJI_INTERPRETER:
					push(this);
					break;

				case DICTIONARY_NEW:
					push(new FijiDictionary());
					break;

				case DICTIONARY_FETCH:
					p = (FijiPointer)pop();
					push(p.locals);
					break;

				case DICTIONARY_STORE:
					p = (FijiPointer)pop();
					o = pop();
					p.locals = (FijiDictionary)o;
					break;

				case INTERNAL_LOCAL:
					i = ((Integer)pop()).intValue();
					o = pop();
					if (i != 0)
						{
						if (doesProgram != null)
							p = doesProgram;
						else
							p = newWord.PFA;

						if (p.locals == null)
							{
							p.locals = new FijiDictionary();
							p.localCount = 1;
							}
						else
							p.localCount++;

						s = (String)o;
						w = new FijiWord(s);
						w.CFA = doLocal;
						w.PFA = new FijiPointer();
						w.PFA.bytes = new byte[1];
						w.PFA.bytes[0] = EXIT;
						w.PFA.objects = new Object[1];
						w.PFA.objects[0] = new Integer(p.localCount);

						p.locals.add(w);
						}
					break;

				case MOST_RECENT_WORD:
					push(newWord);
					break;

				case REFLECT_MAKE_CONSTRUCTOR:
					{
					String className = pop().toString();
					stopAt = ((Number)pop()).intValue();
					String arguments[] = new String[stopAt];
					for (i = 0; i < stopAt; i++)
						{
						arguments[i] = pop().toString();
						}
					FijiReflectConstructor constructor = new FijiReflectConstructor(className, arguments);
					push(constructor);
					push(constructor.worked() ? 1 : 0);
					}
					break;

				case REFLECT_MAKE_METHOD:
					{
					String className = pop().toString();
					String methodName = pop().toString();
					String returnClassName = pop().toString();

					stopAt = ((Number)pop()).intValue();
					String arguments[] = new String[stopAt];
					for (i = 0; i < stopAt; i++)
						{
						arguments[i] = pop().toString();
						}
					FijiReflectMethod method = new FijiReflectMethod(className, methodName, returnClassName, arguments);
					push(method);
					push(method.worked() ? 1 : 0);
					}
					break;

				case REFLECT:
if (false)
					{
					String methodReturnType = (String)pop();
					FijiPointer fijiMethodArgumentTypes = ensureFinal((FijiPointer)pop());
					String[] methodArgumentTypes;
					String methodName = (String)pop();
					FijiPointer constructorArguments = ensureFinal((FijiPointer)pop());
					String className = (String)pop();

					if (fijiMethodArgumentTypes.objects.length == 0)
						methodArgumentTypes = null;
					else
						{
						int length = fijiMethodArgumentTypes.objects.length;
						methodArgumentTypes = new String[length];
						System.arraycopy(fijiMethodArgumentTypes.objects, 0, methodArgumentTypes, 0, length);
						}

					Object[] cArgs;
//					if (constructorArguments.objects.length == 0)
//						cArgs = null;
//					else
						cArgs = constructorArguments.objects;

					if (methodReturnType == "java.lang.Void")
						methodReturnType = null;

					o = new FijiReflect(
						className,
						cArgs,
						methodName,
						methodArgumentTypes,
						methodReturnType);

					if (o != null)
						{
						push(o);
						push(1);
						}
					else
						{
						push(doVariable);
						push(0);
						}
					}
					break;

				default:
					println("unhandled bytecode: " + program.bytes[program.byteX - 1]);
				}
			}
		}


	static String padString = "                                        ";
	static int padStringLength = padString.length();
	String padString(String s, int length)
		{
		int padlength = (length - s.length());
		if (padlength > padStringLength)
			s = padString(s, padlength - padStringLength);
		if (padlength > 0)
			s = padString.substring(0, length - s.length()) + s;
		return s;
		}

	void dumpWordProgram(FijiPointer p, boolean smartDisplay)
		{
		int objectX = p.objectX;
		int ObjectY = objectX;
		boolean wasCompileByte = false;
		for (int i = p.byteX; i < p.bytes.length; i++)
			{
			println(" |    " 
				+ ((wasCompileByte) ? "   " : "")
				+ padString("b" + Integer.toString(i), 4)
				+ ": " + padString(Byte.toString(p.bytes[i]), 4)
				+ " = " + byteToString[p.bytes[i] + 128]
				);
			if (wasCompileByte)
				wasCompileByte = false;
			else
				ObjectY += byteArgumentCount[p.bytes[i] + 128];
			if (smartDisplay)
				{
				for ( ; objectX < ObjectY; objectX++)
					{
					println(" |         o" + objectX + ": "
						+ ((p.objects[objectX] == null) ? "null!"
							: (p.objects[objectX].getClass().getName() 
							   + " = \"" + p.objects[objectX].toString() + "\""
							   )
							)
						);
					}
				}
			wasCompileByte =
				smartDisplay
				&& ((p.bytes[i] == COMPILE_BYTE)
					|| (p.bytes[i] == HANDLE_POSTPONED_BYTECODE));
			}
		if (objectX < p.objects.length)
			{
			if (smartDisplay)
				println(" |   Unused objects (!):");
			else
				println(" |   Objects:");

			for (int i = objectX; i < p.objects.length; i++)
				{
				println(" |      o" + i + ": "
					+ p.objects[i].getClass().getName()
					+ " = \"" + p.objects[i].toString() + "\"");
				}
			}
		}


	Object getObject(String s)
		{
//		int TryType = BYTE;
		boolean KeepGoing = true;
		int radix = 10;
		boolean negative;
//		Object rv = null;

		// rip the number apart
		if (s.charAt(0) == '-')
			{
			s = s.substring(1);
			negative = true;
			}
		else
			negative = false;

		if (s.charAt(0) == '0')
			{
			switch (s.charAt(1))
				{
				case 'x':
					radix = 16;
					s = s.substring(2);
					break;
				case 'b':
					radix = 2;
					s = s.substring(2);
					break;
				case 'o':
					radix = 8;
					s = s.substring(2);
					break;
				}
			}

		if (negative)
			s = '-' + s;

		boolean clip = true;
		char c = s.charAt(s.length() - 1);

		switch (c)
			{
			case 'B':
			case 'C':
			case 'S':
			case 'I':
			case 'L':
			case 'F':
			case 'D':
				break;

			default:
				clip = false;
				c = '.'; // magic cookie for "try int then double"
				break;
			}

		if (clip)
			s = s.substring(0, s.length() - 1);

		switch (c)
			{
			case 'B': // byte
				try
					{
					return Byte.valueOf(s, radix);
					}
				catch (NumberFormatException e)
					{
					}
				// intentional fall-through

			case 'C': // character
				try
					{
					return new Character( (char) (Integer.parseInt(s, radix)) );
					}
				catch (NumberFormatException e)
					{
					}
				// intentional fall-through

			case 'S': // short
				try
					{
					return Short.valueOf(s, radix);
					}
				catch (NumberFormatException e)
					{
					}
				// intentional fall-through

			case '.': // int then double
			case 'I': // int
				try
					{
					return Integer.valueOf(s, radix);
					}
				catch (NumberFormatException e)
					{
					}

				if (c == '.')
					{
					try
						{
						return new Double(s);
						}
					catch (NumberFormatException e)
						{
						}
					return null;
					}
				// intentional fall-through

			case 'L': // long
				try
					{
					return Long.valueOf(s, radix);
					}
				catch (NumberFormatException e)
					{
					}
				// intentional fall-through

			case 'F': // float
				try
					{
					return new Float(s);
					}
				catch (NumberFormatException e)
					{
					}
				// intentional fall-through

			case 'D': // double
				try
					{
					return new Double(s);
					}
				catch (NumberFormatException e)
					{
					}
				break;
			}

		return null;
		}


	StringTokenizer t;
	String pushedString;
	Vector prefixVector;

	boolean moreTokens()
		{
		return t.hasMoreTokens() || (pushedString != null);
		}


	// returns non-null only if s _contains_ the prefix but doesn't exactly match it
	// e.g.  "//s' matches the prefix "//", but "//" doesn't match any prefix at all
	String getPrefix(String s)
		{
		// println("getPrefix: got \"" + s + "\"");
		if (prefixVector == null)
			{
			prefixVector = new Vector();
			prefixVector.addElement("\"");	// "	push string constant
			prefixVector.addElement(".\"");	// ."	print string
			prefixVector.addElement("\\");	// \	comment
			prefixVector.addElement("//");	// //	comment
			prefixVector.addElement("(");	// (	comment
			}

		String t;
		for (int i = 0; i < prefixVector.size(); i++)
			{
			t = (String)prefixVector.elementAt(i);

			// println("getPrefix: is it \"" + t + "\"?");

			// make this >= if you want "//" to be found as a prefix
			if ( (s.length() > t.length())
				 && (s.substring(0, t.length()).compareTo(t) == 0) )
				{
				// println("getPrefix: yep");
				return t;
				}
			}
		// println("getPrefix: none of the above");

		return null;
		}


	void putBackToken(String s)
		{
		pushedString = s;
		}


	String nextToken()
		{
		String s;
		if (pushedString != null)
			{
			s = pushedString;
			pushedString = null;
			}
		else
			{
			if (!t.hasMoreTokens())
				return null;

			s = t.nextToken();
			}
		
		return s;
		}

	String splitOnPrefix(String s)
		{
		String prefix;
		if ((prefix = getPrefix(s)) != null)
			{
			pushedString = s.substring(prefix.length());
			return prefix;
			}
		return null;
		}

	String nextNonWhiteToken()
		{
		String s;
		while (moreTokens())
			{
			s = nextToken();
			// ignore whitespace strings here
			if (s.trim().length() == 0)
				continue;
			return s;
			}

		return null;
		}

	void inlineWord(FijiPointer p)
		{
		int byteX = p.byteX;
		int objectX = p.objectX;
		int i;
		int byteDelta = bytes.size();
		int objectDelta = objects.size();
		boolean notExit = true;

		for (byteX = 0; notExit && (byteX < p.bytes.length); byteX++)
			{
			switch (p.bytes[byteX])
				{
				case EXIT:
					notExit = false;
					break;

				case JUMP:
				case JUMP_CHECK:
				case BRANCH0:
				case BRANCH0_CHECK:
					objects.addElement(new Integer(byteDelta
						+ ((Integer)p.objects[objectX++]).intValue()));
					objects.addElement(new Integer(objectDelta
						+ ((Integer)p.objects[objectX++]).intValue()));
					break;

				default:
					i = byteArgumentCount[p.bytes[byteX] + 128];
					for (; i > 0; i--)
						objects.addElement(p.objects[objectX++]);
					break;
				}
			if (notExit)
				bytes.addElement(new Byte(p.bytes[byteX]));
			}

		for (; objectX < p.objects.length; objectX++)
			objects.addElement(p.objects[objectX++]);
		}


	FijiPointer handleProgram = new FijiPointer(new byte[] { 0, INTERNAL_STOP }, new Object[1]);

	void handleNow(byte b, Object o)
		{
		if (postponeNext)
			{
			postponeNext = false;
			handleLater(b, o, null);
			return;
			}

		handleProgram.bytes[0] = b;
		handleProgram.byteX = 0;
		if (o != null)
			{
			handleProgram.objects[0] = o;
			handleProgram.objectX = 0;
			}
		interpret(handleProgram);
		// this lets us call handleNow() recursively!
		// it hurts to think why.  --lch
		handleProgram.byteX = 1;
		}

	void handleLater(byte b, Object o1, Object o2)
		{
		if (postponeNext)
			{
			postponeNext = false;
			if (b == CALL)
				handleLater(HANDLE_POSTPONED_WORD, o1, null);
			else
				{
				handleLater(HANDLE_POSTPONED_BYTECODE, null, null);
				handleLater(b, null, null);
				}
			return;
			}

		// inline if appropriate
		if ((b == CALL) && (o1 != null))
			{
			FijiWord w = (FijiWord)o1;
			if ((w.inline) && (w.CFA instanceof FijiDoColon))
				{
				inlineWord(w.PFA);
				return;
				}
			}

		bytes.addElement(new Byte(b));
		if (o1 != null)
			{
			objects.addElement(o1);
			if (o2 != null)
				objects.addElement(o2);
			}
		}



	void handleObject(Object o)
		{
		if (!compiling)
			handleNow(PUSH, o);
		else
			handleLater(PUSH, o, null);
		}

	void handleWord(FijiWord word)
		{
		if (!compiling || word.immediate)
			handleNow(CALL, word);
		else
			handleLater(CALL, word, null);
		}

	void handleByteCode(byte b)
		{
		if (!compiling || getByteCodeImmediate(b))
			handleNow(b, null);
		else
			handleLater(b, null, null);
		}


	boolean interpretToken(String s)
		{
		FijiWord word;
		byte byteCode;
		if ((word = lookup(s)) != null)
 			{
			handleWord(word);
			return true;
			}
		else if ((byteCode = getByteCode(s)) != INVALID)
			{
			handleByteCode(byteCode);
			return true;
			}
		return false;
		}


	public void interpret(String text)
		{
		StringTokenizer oldT = t;
		t = new StringTokenizer(text, " \t\r\n", true);

		String s, prefix;
		FijiWord word;
		Object object;

		while ((s = nextNonWhiteToken()) != null)
			{
			if (debugPrintTrace)
				print("<<" + s + ">>");
			if (!interpretToken(s))
				{
				prefix = splitOnPrefix(s);
				if ((prefix == null) || !interpretToken(prefix))
					{
					// the "object runner"
					if ((object = getObject(s)) != null)
						{
						handleObject(object);
						}
					else
						{
						println("got unknown word: \"" + s + "\"");
						}
					}
				}
			}

		t = oldT;
		}


	public void interpret(Reader r)
		{
		BufferedReader br = new BufferedReader(r);

		if (br == null)
			{
			println("error: could not create buffered reader from \"" + r.toString() + "\"");
			return;
			}

		String tarball = "";
		String line;

		try
			{
			for (;;)
				{
				line = br.readLine();
				if (line == null)
					break;
				tarball += "\n" + line;
				}
			}
		catch (IOException e)
			{
			}

		if (debugPrintTrace)
			println("*** interpreting: \n" + tarball + "\n***\n");

		interpret(tarball);
		br = null;
		}


	public void interpretFile(String filename)
		{
		FileReader fr;
		try
			{
			fr = new FileReader(filename);
			}
		catch (FileNotFoundException e)
			{
			println("error: could not open " + filename + "\"");
			return;
			}

		interpret(fr);
		fr = null;
		}


	public static void main(String[] argv)
		{
		if (argv.length == 0)
			{
			println("Fiji: Forth Interpreter In Java  by Larry Hastings");
			println("");
			println("usage: fiji [ forthscript [ forthscript2 ... ] ]");
			println("runs all scripts on the command line, then drops into interactive mode.");
			println("in interactive mode, the prompt is \"ok\".");
			println("to get out of interactive mode, type");
			println("\tbye");
			println("and press Enter.");
			println("");
			}

		FijiInterpreter i = new FijiInterpreter();
		FijiReflect r;

if (false)
	{
		FijiWord javaAddFour = new FijiWord("javaaddfour");
		javaAddFour.CFA = new FijiPlusFour();
		i.dictionary.add(javaAddFour);

		Object[] constructorObjects = new Object[1];
		constructorObjects[0] = new Integer(420);

		String[] objects = new String[1];
		objects[0] = "java.lang.Integer";

		r = new FijiReflect(
			"FijiPlusN",
			constructorObjects,
			"plus",
			objects,
			"java.lang.Integer");

		FijiWord javaAddFourTwenty = new FijiWord("javaaddfourtwenty");
		javaAddFourTwenty.CFA = r;
		i.dictionary.add(javaAddFourTwenty);

		Object[] constructorObjects2 = new Object[1];
		constructorObjects2[0] = new Integer(17);

		String[] objects2 = new String[3];
		objects2[0] = "java.lang.Integer";
		objects2[1] = "java.lang.Integer";
		objects2[2] = "java.lang.String";

		r = new FijiReflect(
			"FijiPlusN",
			constructorObjects2,
			"twoTimesAndPrint",
			objects2,
			"java.lang.Integer");
			
		FijiWord javaTwoTimesAndPrint = new FijiWord("javamultiplyplus17andprint");
		javaTwoTimesAndPrint.CFA = r;
		i.dictionary.add(javaTwoTimesAndPrint);
	}
		
		for (int argvX = 0; argvX < argv.length; argvX++)
			{
			i.interpretFile(argv[argvX]);
			}

		BufferedReader br = new BufferedReader(new InputStreamReader(System.in));
		String s;

		try
			{
			while (!i.bye())
				{
				println("ok");
				s = br.readLine();
				i.interpret(s);
				}
			}
		catch (IOException e)
			{
			}
		}
	}
