// BarcoPower.cpp : Defines the entry point for the application.
//
#include "stdafx.h"

#define DEFAULT_PORT "COM1:"
#define DEFAULT_MODE "9600,n,8,1"
#define DEFAULT_LOG  "C:\\temp\\BarcoPower.log"

#define CMD_STANDBY		0x0E
#define CMD_READ_STATUS	0x4B

/*
READ_STATUS 4BH
This command asks the projector to return the status in which it is working.
	Information sent : Dat1=0, Dat2=0, Dat3=0, Dat4=0

	Returned information is :
		Dat1 :	Bit 0..5 : menu nr. (internal ref. Nr. of displayed menu)
				bit 6 : Green convergence 1 = Installed
				bit 7 : Fast / Slow 1 = Fast
		Dat2 :	cursor position
		Dat3 :	bit 0 : Standby 1 = standby
				bit 1 : Pause 1 = pause
				bit 2 : Text on/off 1 = text on
				bit 3 : PC mode 1 = PC mode
				bit 4 : Sound available
				bit 5 : see below
				bit 6 : see below
				bit 7 : if set, following additional info is available :
					Dat3 :	bit 6 : focus is magnetic
							bit 5 : convergence is stripped
					Dat4 :	bit 0 : orbiting is installed
							bit 1 : soft edge is installed
							bit 2 : contrast modulation is installed
							bit 3 : NS is mounted on the convergence
							bit 4 : controller with ASIC
							bit 5 : IRIS is installed
							bit 6 : dynamic stigmators
							bit 7 : reserved
*/

char *LogFile = DEFAULT_LOG;

/*
 * Log an error message.
 */
void
Error(char *msg)
{
	FILE *fp;
	time_t now;
	char *timestr;

	time(&now);
	timestr = ctime(&now);

	fp = fopen(LogFile, "a");
	if (fp == NULL)
		return;
	fputs(timestr, fp);
	putc(' ', fp);
	fputs(msg, fp);
	putc('\n', fp);
	fclose(fp);
}


/*
 * Add a byte to an outgoing projector command, quoting if needed.
 *
 * Input:
 *	byte	Byte to add.
 *	buf		Start of buffer.
 *	len		Pointer to number of bytes currently used in buffer.
 *	size	Maximum size of buffer.
 *
 * Return:
 *	0		Not enough space.
 *	1		Byte added.
 */
int
AddByte(BYTE byte, BYTE *buf, DWORD *len, DWORD size)
{
	if (byte == 0x80 || byte == 0xff || byte == 0xfe)
	{
		if (*len + 1 >= size)
			return 0;
		buf[(*len)++] = 0x80;
		buf[(*len)++] = byte & 0x7f;
		return 1;
	}

	if (*len >= size)
		return 0;
	buf[(*len)++] = byte;
	return 1;
}

/*
 * Send a command to the projector.
 *
 * Input:
 *	hComm	Handle to serial port.
 *	address	Projector address (0-255).
 *	code	1-byte command code to send.
 *	data	Data byte pointer.  NULL = no data bytes.
 *	count	Number of data bytes.
 *
 * Return:
 *	0		Failure.
 *	1		Success.
 */
int
SendCommand(HANDLE hComm, BYTE address, BYTE code, BYTE *data, int count)
{
	BYTE	*buffer;
	DWORD	nBytes = 0;
	DWORD	size = count * 2 + 6;
	int		i;
	int		checksum = address + code;
	DWORD	bytesWritten;

	// Allocate enough buffer space to quote all the data bytes.
	buffer = (BYTE *) malloc(size);
	if (buffer == NULL)
	{
		Error("Can't allocate buffer for outgoing command");
		return (0);
	}

	buffer[nBytes++] = 2;        // Start of command frame
	buffer[nBytes++] = address == 1 ? 0 : 1;
	buffer[nBytes++] = address;
	buffer[nBytes++] = code;

	for (i = 0; i < count; i++)
	{
		buffer[nBytes++] = data[i];
		checksum += data[i];
	}

	buffer[nBytes++] = checksum & 0xff;

	if (! WriteFile(hComm, buffer, nBytes, &bytesWritten, NULL) ||
		bytesWritten < nBytes)
	{
		Error("Timeout sending command to projector");
		return (0);
	}

	free(buffer);
	return 1;
}


/*
 * Send raw bytes to the projector and display the result.
 */
void
DoRaw(HANDLE hComm)
{
	BYTE byte;
	DWORD bytesWritten, bytesRead;
	int b;
	char *token;
	char buf[1000];

	buf[0] = '\0';
	while ((token = strtok(NULL, "\t ")) != NULL)
	{
		sscanf(token, "%x", &b);
		byte = (BYTE)b;
		WriteFile(hComm, &byte, 1, &bytesWritten, NULL);
		sprintf(buf + strlen(buf), "%02x ", byte);
	}

	strcat(buf, "\n");

	while (ReadFile(hComm, &byte, 1, &bytesRead, NULL) &&
		   bytesRead == 1)
	{
		sprintf(buf + strlen(buf), "%02x ", byte);
	}

	MessageBox(NULL, buf, "Result bytes", MB_OK);
}


/*
 * Receive a response from the projector.
 *
 * Input:
 *	hComm	Handle to serial port.
 *	address	Buffer for projector address.
 *	code	Buffer for command code.
 *	data	Buffer for data bytes.
 *	count	Buffer for number of bytes read.
 *	size	Size of data buffer.
 *
 * Return:
 *	0		Failure.
 *	1		Success.
 */
int
ReceiveResponse(HANDLE hComm, BYTE *address, BYTE *code, BYTE *data, DWORD *count, DWORD size)
{
	BYTE	byte;
	DWORD	bytesRead;
	DWORD	checksum, i;

	// Read until we hit a start-of-frame marker.
	do {
		bytesRead = 0;
		if (! ReadFile(hComm, &byte, 1, &bytesRead, NULL) ||
			bytesRead < 1)
		{
			Error("Failed to read from projector");
			return (0);
		}

		if (byte != 6)
		{
			char msg[50];
			sprintf(msg, "Got unexpected byte %02x", byte);
			Error(msg);
		}
	} while (byte != 6);

	if (! ReadFile(hComm, address, 1, &bytesRead, NULL) ||
		bytesRead < 1)
	{
		Error("Failed to read address from projector");
		return (0);
	}

	if (! ReadFile(hComm, code, 1, &bytesRead, NULL) ||
		bytesRead < 1)
	{
		Error("Failed to read address from projector");
		return (0);
	}

	checksum = *code + *address;

	if (! ReadFile(hComm, data, size, count, NULL) ||
		*count < size)
	{
		Error("Incomplete data part of response");
		return (0);
	}

	for (i = 0; i < size; i++)
		checksum += data[i];

	if (! ReadFile(hComm, &byte, 1, &bytesRead, NULL) ||
		bytesRead < 1)
	{
		Error("Failed to read checksum");
		return (0);
	}

	if ((checksum & 0xff) != byte)
	{
		Error("Response checksum didn't match");
		return (0);
	}

	return (1);
}



int APIENTRY WinMain(HINSTANCE hInstance,
                     HINSTANCE hPrevInstance,
                     LPSTR     lpCmdLine,
                     int       nCmdShow)
{
	DCB dcb;
	LPSTR token;
	HANDLE hComm;
	LPSTR portName = DEFAULT_PORT;
	COMMTIMEOUTS timeouts;
	char mode[100];
	int desiredPower = -1;
	int timeout = 3000;
	BYTE address = 1, code, data[4];
	DWORD count;
	int retries = 0;
	int maxRetries = 3;

	strcpy(mode, DEFAULT_MODE);

	token = strtok(lpCmdLine, " \t");
	while (token != NULL)
	{
		if (! stricmp(token, "/port"))
		{
			token = strtok(NULL, " \t");
			if (token == NULL)
			{
				Error("No port name specified");
				return(1);
			}

			portName = token;
		}
		else if (! stricmp(token, "/addr"))
		{
			token = strtok(NULL, " \t");
			if (token == NULL)
			{
				Error("No projector address specified");
				return(1);
			}

			address = atoi(token);
		}
		else if (! stricmp(token, "/mode"))
		{
			token = strtok(NULL, " \t");
			if (token == NULL)
			{
				Error("No mode specified");
				return(1);
			}

			strncpy(mode, token, sizeof(mode)-1);
			mode[sizeof(mode)-1] = '\0';
		}
		else if (! stricmp(token, "/timeout"))
		{
			token = strtok(NULL, " \t");
			if (token == NULL)
			{
				Error("No timeout specified");
				return(1);
			}

			timeout = atoi(token);
		}
		else if (! stricmp(token, "/log"))
		{
			token = strtok(NULL, " \t");
			if (token == NULL)
			{
				Error("No log specified");
				return(1);
			}

			LogFile = token;
		}
		else if (! stricmp(token, "/on"))
		{
			desiredPower = 1;
		}
		else if (! stricmp(token, "/off"))
		{
			desiredPower = 0;
		}
		else if (! stricmp(token, "/raw"))
		{
			desiredPower = 2;
			break;
		}
		else
		{
			Error("Usage: barcopower [/port COMx:] [/addr <projector-address>]\n"
				"          [/mode baud,bits,parity,stop] [/log filename]\n"
				"          [/timeout msec] {/on|/off|/raw <hex bytes to send>}\n");
			return(1);
		}

		token = strtok(NULL, " \t");
	}

	if (desiredPower < 0)
	{
		Error("Must specify /on or /off");
		return(1);
	}

	hComm = CreateFile( portName,  
                    GENERIC_READ | GENERIC_WRITE, 
                    0,0, 
                    OPEN_EXISTING,
                    0,0);
	if (hComm == INVALID_HANDLE_VALUE)
	{
		char message[1000];

		strcpy(message, "Can't open port ");
		strcat(message, portName);
		Error(message);
		return(1);
	}

	// Now configure the serial port to talk to the projector.
	FillMemory(&dcb, sizeof(dcb), 0);
	dcb.DCBlength = sizeof(dcb);
	if (! BuildCommDCB(mode, &dcb))
	{
		Error("Can't build DCB to configure serial port");
		CloseHandle(hComm);
		return (1);
	}

	if (! SetCommState(hComm, &dcb))
	{
		Error("Can't configure serial port");
		CloseHandle(hComm);
		return (1);
	}

	// Set some reasonable timeouts so we don't hang if the projector doesn't answer.
	timeouts.ReadIntervalTimeout = 1000; 
	timeouts.ReadTotalTimeoutMultiplier = 0;
	timeouts.ReadTotalTimeoutConstant = timeout;
	timeouts.WriteTotalTimeoutMultiplier = 0;
	timeouts.WriteTotalTimeoutConstant = 3000;

	if (!SetCommTimeouts(hComm, &timeouts))
	{
		Error("Can't set comm port timeouts");
		CloseHandle(hComm);
		return (1);
	}

	if (desiredPower == 2)
	{
		DoRaw(hComm);
	}
	else
	{
		do {
			if (! SendCommand(hComm, address, CMD_READ_STATUS, (unsigned char *)"\0\0\0", 4))
			{
				CloseHandle(hComm);
				return (1);
			}
		} while (! ReceiveResponse(hComm, &address, &code, data, &count, sizeof(data)) &&
			     ++retries < maxRetries);

		if (retries >= maxRetries)
		{
			CloseHandle(hComm);
			return (1);
		}

		// Bit 0 of the third byte is set if we're in standby mode.
		if (desiredPower == 1 && (data[2] & 1) == 1)
		{
			SendCommand(hComm, address, CMD_STANDBY, (unsigned char *)"\0\0\0", 4);
		}
		else if (desiredPower == 0 && (data[2] & 1) == 0)
		{
			SendCommand(hComm, address, CMD_STANDBY, (unsigned char *)"\0\0\0", 4);
		}
	}

	CloseHandle(hComm);
	return 0;
}
