/*
 * Functions for dealing with UDP datagrams.
 */
#include <stdlib.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <errno.h>
#include "calleridsentryudp.h"
#include "callerid.h"

/*
 * Listen to a UDP port at a specified IP address.
 *
 * Input:
 *	addr	Address to listen to.  Can be INADDR_ANY to listen on all
 *		interfaces.
 *	port	Port number to listen to.
 *
 * Returns:
 *	File descriptor of listening socket, or -1 on error (errno will be
 *	set).
 */
int
serversock_udp(struct in_addr *addr, int port)
{
	struct sockaddr_in	sin;
	int			sockfd;
	struct in_addr		any;

	sockfd = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP);
	if (sockfd < 0)
		return -1;

	if (addr == NULL)
	{
		any.s_addr = INADDR_ANY;
		addr = &any;
	}

	sin.sin_family = AF_INET;
	memcpy(&sin.sin_addr, addr, sizeof(sin.sin_addr));
	sin.sin_port = htons(port);

	if (bind(sockfd, (struct sockaddr *)&sin, sizeof(sin)) < 0)
	{
		int	errno_old = errno;	// in case close() resets it

		close(sockfd);
		errno = errno_old;
		return -1;
	}

	return sockfd;
}


/*
 * Sends a datagram to a remote host.
 *
 * Input:
 *	sock	Socket to use to send, or -1 to allocate a temporary one.
 *	addr	Remote host to send to.
 *	port	Remote port to send to.
 *	buf	Data to send.
 *	len	Number of bytes to send.
 *
 * Returns:
 *	0 on success, -1 on error (errno will be set).
 */
int
send_udp(int sock, struct in_addr *addr, int port, void *buf, int len)
{
	struct sockaddr_in	sin;
	int			sockfd = sock;
	int			result;
	int			errno_old;

	if (sockfd < 0)
	{
		// Create a generic UDP socket bound to no port in particular
		sin.sin_addr.s_addr = INADDR_ANY;
		sockfd = serversock_udp(&sin.sin_addr, 0);
		if (sockfd < 0)
			return -1;
	}

	sin.sin_family = AF_INET;
	memcpy(&sin.sin_addr, addr, sizeof(sin.sin_addr));
	sin.sin_port = htons(port);

	result = sendto(sockfd, buf, len, 0, (struct sockaddr *)&sin,
			sizeof(sin));

	errno_old = errno;
	if (sockfd != sock)
		close(sockfd);
	errno = errno_old;

	return result;
}


/*
 * Sends a packet of caller ID information to a client.
 *
 * Input:
 *	sock	Socket to use to send, or -1 to allocate a temporary one.
 *	addr	Remote host to send to.
 *	port	Remote port to send to.
 *	name	Caller's name.
 *	phone	Caller's phone number.
 */
void
send_caller_info(int sock,
		 struct in_addr *addr,
		 int port,
		 char *name,
		 char *phone)
{
	struct CallerIDDatagram packet;

	if (name == NULL || ! strcmp(name, "o"))
		name = "Unknown";
	if (phone == NULL || ! strcmp(phone, "o"))
		phone = "Unknown";

	memset(&packet, 0, sizeof(packet));
	packet.PacketType = CID_TYPE_INCOMING_CALL;
	strncpy(packet.Param2, name, sizeof(packet.Param2));
	strncpy(packet.Param3, phone, sizeof(packet.Param3));

	send_udp(sock, addr, port, &packet, sizeof(packet));
}


/*
 * Sends a subscribe request to a server.
 *
 * Input:
 *	sock	Socket to use to send.
 *	addr	Remote host to send to.
 *	port	Remote port to send to.
 *	localport Our port number to tell to server.
 */
void
send_subscribe(int sock, struct in_addr *addr, int port, int localport)
{
	struct CallerIDDatagram packet;

	memset(&packet, 0, sizeof(packet));
	packet.PacketType = CID_TYPE_SUBSCRIBE;
	packet.Param1 = localport;

	send_udp(sock, addr, port, &packet, sizeof(packet));
}


/*
 * Translates an incoming call packet into our internal structure and
 * displays a popup if desired.
 *
 * Input:
 *	Packet to decode
 */
void
incoming_call(struct CallerIDDatagram *packet)
{
	CALL	call;

	memset(&call, 0, sizeof(call));
	strncpy(call.name, packet->Param2, sizeof(call.name));
	strncpy(call.number, packet->Param3, sizeof(call.number));
	call.got_name = 1;
	call.got_number = 1;

	handle_call(-1, &call);
}


/*
 * Reads and parses a client request.
 *
 * Input:
 *	sock	Socket to receive from.
 */
void
handle_socket(int sock)
{
	struct CallerIDDatagram	packet;
	struct sockaddr_in	addr;
	size_t			addr_len = sizeof(addr);
	int			bytes;

	bytes = recvfrom(sock, &packet, sizeof(packet), 0,
				(struct sockaddr *)&addr, &addr_len);
	if (bytes <= 0)
		return;

	switch (packet.PacketType) {
	case CID_TYPE_SUBSCRIBE:
		if (verbose)
			printf("Subscribing %s:%d\n",
				inet_ntoa(addr.sin_addr), ntohs(addr.sin_port));
		add_client(&addr.sin_addr, ntohs(addr.sin_port));
		break;

	case CID_TYPE_UNSUBSCRIBE:
		if (verbose)
			printf("Unsubscribing %s:%d\n",
				inet_ntoa(addr.sin_addr), ntohs(addr.sin_port));
		remove_client(&addr.sin_addr, ntohs(addr.sin_port));
		break;

	case CID_TYPE_INCOMING_CALL:
		if (verbose)
			printf("Incoming call from server %s:%d\n",
				inet_ntoa(addr.sin_addr), ntohs(addr.sin_port));
		incoming_call(&packet);
		break;

	case CID_TYPE_TEST:
		sendto(sock, &packet, sizeof(packet), 0,
			(struct sockaddr *)&addr, addr_len);
		break;
	}
}
