/*
 * Maintains a database of name to phone number mappings.
 */
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include "gfuncs.h"

#define HASH_SIZE	71

#define IS_PHONE_DIGIT(x)  (isdigit(x) || (x) == '*' || (x) == '#')

typedef struct _phonenumber PHONENUMBER;
struct _phonenumber {
	char	*phone;
	char	*name;
	PHONENUMBER *next;
};

/* Hashtable of phonenumbers. */
static PHONENUMBER *mappings[HASH_SIZE];

/*
 * Computes the hash value for a phone number.  The hash function is really
 * simple but sufficient for the data set.
 */
static int
hash(char *phone)
{
	int	value = 0;

	for (; *phone; phone++)
	{
		if (IS_PHONE_DIGIT(*phone))
		{
			value = (value << 1) | *phone;
			if (value & 0x800000)
				value ^= 0xa53e4d;
		}
	}
	
	return value % HASH_SIZE;
}


/*
 * Compares two phone numbers, ignoring non-phonenumber characters.
 */
static int
phonecmp(char *n1, char *n2)
{
	while (1)
	{
		while (*n1 != '\0' && ! IS_PHONE_DIGIT(*n1))
			n1++;
		while (*n2 != '\0' && ! IS_PHONE_DIGIT(*n2))
			n2++;
		if (*n1 > *n2)
			return -1;
		if (*n2 > *n1)
			return 1;
		if (*n1 == '\0')
			return 0;
		n1++;
		n2++;
	}
}


/*
 * Adds a phone number mapping to the hashtable.
 */
void
add_phone(char *phone, char *name)
{
	char	*s, *t;
	PHONENUMBER **bucket, *entry;

	entry = gcalloc(sizeof(PHONENUMBER));
	entry->name = gstrdup(name);
	entry->phone = gstrdup(phone);

	bucket = &mappings[hash(phone)];
	entry->next = *bucket;
	*bucket = entry;
}


/*
 * Looks up a phone number in the hashtable.
 *
 * Returns:
 *	Name associated with number, or NULL if no mapping was found.
 */
char *
lookup_name(char *phone)
{
	PHONENUMBER *bucket;
	
	for(bucket=mappings[hash(phone)]; bucket != NULL; bucket=bucket->next)
		if (! phonecmp(bucket->phone, phone))
			return bucket->name;
	return NULL;
}


/*
 * Reads a list of phone-to-name mappings from a file.
 *
 * Input:
 *	Filename to read.
 *
 * Returns:
 *	Nonzero if the file couldn't be read.
 */
int
read_phone_file(char *filename)
{
	FILE	*fp;
	char	line[100], *phone, *name;

	fp = fopen(filename, "r");
	if (fp == NULL)
		return -1;
	
	while (fgets(line, sizeof(line), fp) != NULL)
	{
		if (line[0] == ';')
			continue;

		phone = strtok(line, "\t\r\n ");
		if (phone == NULL)
			continue;
		name = strtok(NULL, "\r\n");
		if (name == NULL)
			continue;

		add_phone(phone, name);
	}

	fclose(fp);
	return 0;
}


/*
 * Initializes the in-memory phone number mapping table.
 */
void
init_phone_table()
{
	memset(&mappings[0], 0, sizeof(mappings));
}
