/*
 * Maintain the list of caller ID clients.
 */
#include <stdio.h>
#include <netinet/in.h>
#include <malloc.h>

typedef struct _client {
	struct _client	*next;
	struct _client	*prev;
	struct in_addr	addr;
	int		port;
} CLIENT;

CLIENT *clients = NULL;

/*
 * Broadcast a caller ID notification to all clients.
 *
 * Input:
 *	sock	Socket to use to send
 *	name	Name to send
 *	number	Phone number to send
 */
void
broadcast(int sock, char *name, char *number)
{
	CLIENT	*c;

	for (c = clients; c != NULL; c = c->next)
		send_caller_info(sock, &c->addr, c->port, name, number);
}


/*
 * Finds a client on the client list.
 *
 * Input:
 *	addr	IP address of client.
 *	port	Port number.
 * Returns:
 *	Client structure address, or NULL if not found.
 */
static CLIENT *
find_client(struct in_addr *addr, int port)
{
	CLIENT	*c;

	for (c = clients; c != NULL; c = c->next)
		if (addr->s_addr == c->addr.s_addr && port == c->port)
			return c;
	return NULL;
}


/*
 * Add a new client to the list.
 *
 * Input:
 *	addr	IP address of client.
 *	port	Port number.
 */
void
add_client(struct in_addr *addr, int port)
{
	CLIENT	*c;

	// No-op if we already know about this client.
	if (find_client(addr, port) != NULL)
		return;

	c = calloc(1, sizeof(CLIENT));
	if (c == NULL)
	{
		perror("Can't add new client");
		exit(1);
	}

	memcpy(&c->addr, addr, sizeof(struct in_addr));
	c->port = port;
	c->next = clients;
	if (clients != NULL)
		clients->prev = c;
	clients = c;
}


/*
 * Remove a client from the list.
 *
 * Input:
 *	addr	IP address of client.
 *	port	Port number.
 */
void
remove_client(struct in_addr *addr, int port)
{
	CLIENT	*c;

	c = find_client(addr, port);
	if (c == NULL)
		return;

	if (c->next != NULL)
		c->next->prev = c->prev;
	if (c->prev != NULL)
		c->prev->next = c->next;
	else
		clients = c->next;

	free(c);
}

