/*
 * $Id: callerid.c,v 1.3 2001/11/30 07:21:34 koreth Exp $
 *
 * $Log: callerid.c,v $
 * Revision 1.3  2001/11/30 07:21:34  koreth
 * Add davinci27's LED and screensaver code.
 * Handle getting the tail end of a command echo during modem initialization.
 * Blank line after caller ID info triggers handle_call, so we can handle
 * caller ID systems that return just a number with no name.
 *
 */
#include <stdio.h>
#include <fcntl.h>
#include <stdlib.h>
#include <sys/modem.h>
#include <errno.h>
#include <sys/select.h>
#include <netdb.h>
#include <netinet/in.h>
#include <sys/time.h>
#include <Pt.h>
#include <Pt.h>
#include "callerid.h"

char *logfile = NULL;
int verbose = 0;
PtWidget_t *window = NULL;


/*
 * Parses input from the modem.  Returns true if we've just received
 * a block of call information.
 */
int
parse_modem_string(char *str, CALL *call)
{
	if (str[0] == '\0')
	{
		if (call->got_name || call->got_number)
			return 1;
		return 0;
	}

	if (! strcmp(str, "ring"))
	{
		memset(call, 0, sizeof(CALL));
	}
	else if (! strncmp(str, "name = ", 7))
	{
		strncpy(call->name, str + 7, sizeof(call->name)-1);
		call->got_name = 1;
	}
	else if (! strncmp(str, "nmbr = ", 7))
	{
		strncpy(call->number, str + 7, sizeof(call->number)-1);
		call->got_number = 1;
	}
	else if (! strncmp(str, "time = ", 7))
	{
		sprintf(call->time, "%2.2s:%2.2s", str + 7, str + 9);
		call->got_time = 1;
	}
	else if (! strncmp(str, "date = ", 7))
	{
		sprintf(call->date, "%2.2s/%2.2s", str + 7, str + 9);
		call->got_date = 1;
	}
	else if (verbose)
		printf("Got unrecognized modem string '%s'\n", str);
	
	return 0;
}


static void send_key( long key )
{
	PhEvent_t event;
	PhKeyEvent_t key_event;
	PhRect_t rect;

	rect.ul.x = rect.ul.y = 1;
	rect.lr.x = rect.lr.y = 1;

	memset( &event , 0, sizeof(event) );
	memset( &key_event, 0, sizeof(key_event) );

	event.type = Ph_EV_KEY;
	event.emitter.rid = Ph_DEV_RID;
	event.num_rects = 1;
	event.data_len = sizeof(key_event);
	event.input_group = 1;

	key_event.key_cap = key;
	key_event.key_sym = key;
	key_event.key_flags = Pk_KF_Sym_Valid | Pk_KF_Cap_Valid | 
	Pk_KF_Key_Down;

	PhEventEmit( &event, &rect, &key_event );
	key_event.key_flags &= ~(Pk_KF_Key_Down | Pk_KF_Sym_Valid);
	PhEventEmit( &event, &rect, &key_event );
	return;
}


void
handle_call(int socketfd, CALL *call)
{
	time_t	now = time(NULL);
	char	*name;
	char	scratch[200];
	char	const *button = "&Close";
	char	Helvetica12[MAX_FONT_TAG];

	if (! (call->got_name || call->got_number))
		return;

	if (! call->got_date)
		strftime(call->date, 6, "%m/%d", localtime(&now));
	if (! call->got_time)
		strftime(call->time, 6, "%H:%M", localtime(&now));

	name = lookup_name(call->number);
	if (name == NULL)
		name = call->name;

	/* Pretty up 10-digit phone numbers. */
	if (strlen(call->number) == 10)
	{
		sprintf(scratch, "(%3.3s) %3.3s-%4.4s",
			call->number, call->number + 3, call->number + 6);
		strcpy(call->number, scratch);
	}

	if (socketfd >= 0)
		broadcast(socketfd, name, call->number);

	sprintf(scratch, "%s %s INCOMING CALL: name '%s' number '%s'\n",
			call->date, call->time, name, call->number);

	if (logfile != NULL)
	{
		FILE	*fp;

		if (strcmp(logfile, "-"))
		{
			/*
			 * Reopen the file each time, rather than holding it
			 * open, so the user can remove or edit the file
			 * in between calls.
			 */
			fp = fopen(logfile, "a");

			if (fp == NULL)
				perror(logfile);
			else
			{
				fputs(scratch, fp);
				fclose(fp);
			}
		}
		else
		{
			fputs(scratch, stdout);
			fflush(stdout);
		}
	}

	if (window != NULL)
	{
		sprintf(scratch, "%s\n%s\n%s %s", name,
			call->number, call->date, call->time);
		send_key(0xF080); //Turn Screen Saver Off
		system("gpio -a 0x40010004 -b6 -m3 -s 3"); //Turn Screen On
		system("SetLEDState 1"); //Blink LED
		switch( PtAlert( NULL, NULL, "Incoming Call", NULL, scratch,  PfGenerateFontName("Helvetica", 0, 12,  Helvetica12),  1, &button, NULL, 1, 3, Pt_MODAL ) ) 
		{
		    case 1:
			/* cancel */
			system("SetLEDState 0"); //Stop Led
			break;
		    case 2:
			break;
		}
	}

	memset(call, 0, sizeof(CALL));
}


int
handle_modem(int modemfd, int socketfd, CALL *call)
{
	int	len;
	char	buf[1000];

	/*
	 * Since modem_read does buffering internally, we need to empty
	 * out its input queue before returning to the main loop's select().
	 */
	while (1)
	{
		len = modem_read(modemfd, buf, sizeof(buf) - 1, 2, 5,
					MODEM_ALLOW8BIT, NULL);
		if (len < 0)
		{
			if (errno == ETIMEDOUT)
				return (0);
			perror("Reading from modem");
			return (-1);
		}

		//printf("Got '%s'\n", buf);
		if (parse_modem_string(buf, call))
			handle_call(socketfd, call);
		return (0);
	}
}


void usage(char *name)
{
	fprintf(stderr, "Usage: %s [-c host] [-f dbfile] [-l file|-] [-m modem] [-p port] [-r n] [-v] [-w]\n", name);
	fprintf(stderr, "\tDefault port is %d\n", DEFAULT_PORT);
	fprintf(stderr, "\tDefault modem is %s\n", DEFAULT_MODEM);
	exit(1);
}


main(int argc, char **argv)
{
	struct in_addr hostaddr;
	struct hostent *hent;
	struct timeval tv, *mytv;
	int	modemfd = -1;
	int	socketfd;
	char	*dbfile = NULL;
	char	*modem = DEFAULT_MODEM;
	char	*host = NULL;
	int	port = DEFAULT_PORT;
	int	resubscribe = 0;
	char	buf[1000];
	int	len;
	CALL	call;
	int	c;

	while ((c = getopt(argc, argv, "f:c:l:m:p:r:vw")) != EOF) {
		switch (c) {
		case 'w':
			if ((window = PtAppInit(NULL, &argc, argv, 0, NULL)) == NULL)
				PtExit(1);
			break;
		case 'f':
			dbfile = optarg;
			break;
		case 'l':
			logfile = optarg;
			break;
		case 'm':
			modem = optarg;
			break;
		case 'p':
			port = atoi(optarg);
			if (port == 0)
				usage(argv[0]);
			break;
		case 'r':
			resubscribe = atoi(optarg);
			break;
		case 'c':
			host = optarg;
			modem = NULL;
			break;
		case 'v':
			verbose = 1;
			break;
		default:
			usage(argv[0]);
		}
	}

	if (resubscribe)
	{
		mytv = &tv;
		tv.tv_usec = 0;
		tv.tv_sec = resubscribe;
	}
	else
		mytv = NULL;

	init_phone_table();

	if (dbfile != NULL)
	{
		if (read_phone_file(dbfile))
		{
			perror(dbfile);
			exit(1);
		}
	}

	socketfd = serversock_udp(NULL, host == NULL ? port : port + 1);
	if (socketfd < 0)
	{
		perror("Opening server socket");
		exit(1);
	}

	if (verbose && host == NULL)
		printf("Listening on port %d\n", port);

	if (modem != NULL)
	{
		modemfd = modem_open(modem, 57600);
		if (modemfd < 0)
		{
			if (host == NULL)
			{
				if (verbose)
					printf("Launching rsmodem\n");
				system("/kojak/rsmodem");
				sleep(1);
				modemfd = modem_open(modem, 57600);
				if (modemfd < 0)
				{
					perror(modem);
					exit(1);
				}
			}
			else
			{
				fprintf(stderr, "No modem, client-only mode\n");
			}
		}
	}

	if (modemfd >= 0)
	{
		if (modem_write(modemfd, "ATS0=0 +VCID=1\\r") < 0)
		{
			perror("Initializing modem");
			close(modemfd);
			exit(1);
		}

		memset(&call, 0, sizeof(call));

		do {
			len = modem_read(modemfd, buf, sizeof(buf) - 1, 2, 15,
						0, NULL);
			if (len < 0)
			{
				perror("Reading init response from modem");
				close(modemfd);
				exit(1);
			}

			if (strchr(buf, '='))	// tail end of an echo
				continue;

			if (buf[0] != '\0')
			{
				if (strcmp(buf, "ok"))
				{
					fprintf(stderr, "Modem response '%s'\n",
						buf);
					close(modemfd);
					exit(1);
				}
				break;
			}
		} while (buf[0] == '\0');

		if (verbose)
			printf("Modem is awaiting calls\n");
	}

	if (host != NULL)
	{
		hent = gethostbyname(host);
		if (hent == NULL)
		{
			fprintf(stderr, "%s: %s\n", host, hstrerror(h_errno));
			exit(1);
		}

		memcpy(&hostaddr, hent->h_addr, sizeof(hostaddr));

		if (verbose)
			printf("Using server %s\n", inet_ntoa(hostaddr));

		send_subscribe(socketfd, &hostaddr, port, port + 1);
	}

	while (1)
	{
		fd_set	rfds, xfds;
		int	nfds = modemfd + 1;
		int	result;

		FD_ZERO(&rfds);
		FD_ZERO(&xfds);
		if (modemfd >= 0)
			FD_SET(modemfd, &rfds);
		if (socketfd >= 0)
			FD_SET(socketfd, &rfds);
		if (socketfd > modemfd)
			nfds = socketfd + 1;

		result = select(nfds, &rfds, &xfds, &xfds, mytv);
		if (result < 0)
		{
			perror("select");
			close(socketfd);
			close(modemfd);
			exit(1);
		}

		if (result == 0 && host != NULL)
		{
			if (verbose)
				printf("Resubscribing\n");
			send_subscribe(socketfd, &hostaddr, port, port + 1);
			continue;
		}

		if (modemfd >= 0 && FD_ISSET(modemfd, &rfds))
		{
			if (handle_modem(modemfd, socketfd, &call))
			{
				close(socketfd);
				close(modemfd);
				exit(1);
			}
		}
		if (socketfd >= 0 && FD_ISSET(socketfd, &rfds))
			handle_socket(socketfd);
	}
}
