/*
 * Prebuffer management module for slow filesystems.
 */
#include "prebuf.h"
#include <pthread.h>
#include <errno.h>
#include <malloc.h>
#include <signal.h>


#define HASH_SIZE	71


/*
 * Prebuffered file information.  We maintain a small hashtable of these.
 * Each file has a circular buffer and a condition variable; when we fill
 * the buffer, we wait for the cond, which is triggered whenever a reader
 * pulls bytes from a full buffer.
 *
 * head == tail means the buffer is empty
 * head == tail + 1 means the buffer is full (yes, this wastes 1 buffer byte)
 */
typedef struct _prebuf PREBUF;
struct _prebuf {
	int		fd;
	char		*buf;
	int		buf_size;
	int		chunk_size;
	int		head;		// Next byte to be read
	int		tail;		// Next byte to be filled
	pthread_cond_t	space_avail_cond; // When bytes are consumed
	pthread_cond_t	data_avail_cond; // When bytes arrive
	pthread_mutex_t lock;
	pthread_t	thread;
	int		interrupted;	// 1 to interrupt read thread
	int		error;		// 1 if we got a read error
	int		eof;		// 1 if we've hit end of file
	int		done;		// 1 if read thread is exiting

	PREBUF		*next;
};

pthread_mutex_t prebuf_lock = PTHREAD_MUTEX_INITIALIZER;
PREBUF	*prebuf_hash[HASH_SIZE];


/* Dummy signal handler so we can interrupt reads. */
void sigusr1(int sig) { }

/*
 * Waits for input from a file.  One of these runs for each prebuffered file.
 *
 * Input:
 *	Prebuffered file to read from.
 */
static void *prebufmain(void *arg)
{
	PREBUF		*pb = (PREBUF *)arg;
	int		bytes_left, bytes_read;

	signal(SIGUSR1, sigusr1);
	pthread_mutex_lock(&pb->lock);

	while (1)
	{
		/*
		 * Wait for space to become available if needed.
		 */
		do {
			/*
			 * How many more bytes can we read?
			 */
			if (pb->tail == pb->buf_size && pb->head > 0)
			{
				pb->tail = 0;
			}

			if (pb->tail < pb->head)
				bytes_left = pb->head - pb->tail - 1;
			else
				bytes_left = pb->buf_size - pb->tail;

			if (bytes_left == 0)
			{
//printf("waiting for space_avail_cond\n");
				pthread_cond_wait(&pb->space_avail_cond,
							&pb->lock);
				if (pb->interrupted)
					break;
//printf("got space_avail_cond\n");
			}
		} while (bytes_left == 0);

		if (bytes_left > pb->chunk_size)
			bytes_left = pb->chunk_size;
		if (pb->head == pb->tail)
			pb->head = pb->tail = 0;

		/*
		 * Fill 'er up.
		 */
		pthread_mutex_unlock(&pb->lock);
		bytes_read = read(pb->fd, pb->buf + pb->tail, bytes_left);

		// Try again just in case, if this looks like EOF.
		if (bytes_read == 0)
			bytes_read = read(pb->fd, pb->buf + pb->tail, bytes_left);

		pthread_mutex_lock(&pb->lock);

//if (bytes_read != bytes_left) printf("fd %d adding %d bytes\n", pb->fd, bytes_read);
		if (pb->interrupted)
			break;

		if (bytes_read < 0)
		{
//printf("fd %d setting err\n", pb->fd);
			pb->error = 1;
			break;
		}

		if (bytes_read == 0)
		{
//printf("fd %d setting EOF\n", pb->fd);
			pb->eof = 1;
			break;
		}

		pb->tail += bytes_read;
		pthread_cond_signal(&pb->data_avail_cond);
	}

//printf("fd %d setting done\n", pb->fd);
	pb->done = 1;
	pthread_cond_signal(&pb->data_avail_cond);
	pthread_mutex_unlock(&pb->lock);
	return NULL;
}


/*
 * Starts prebuffering a file.
 *
 * Input:
 *	File handle to start prebuffering.
 *	Maximum number of bytes to prebuffer.
 * Returns:
 *	0 on success, -1 on failure.
 */
int prebufstart(int fd, int max_bytes, int chunk_size)
{
	int		hashpos = fd % HASH_SIZE;
	pthread_attr_t	attr;
	PREBUF		*prebuf = calloc(1, sizeof(PREBUF));

	if (prebuf == NULL)
		return -1;
	
	prebuf->fd = fd;
	prebuf->buf_size = max_bytes;
	prebuf->chunk_size = chunk_size;
	prebuf->buf = malloc(max_bytes);
	if (prebuf->buf == NULL)
	{
		free(prebuf);
		return -1;
	}

	if (pthread_mutex_init(&prebuf->lock, NULL))
	{
		free(prebuf->buf);
		free(prebuf);
		return -1;
	}

	if (pthread_cond_init(&prebuf->space_avail_cond, NULL))
	{
		pthread_mutex_destroy(&prebuf->lock);
		free(prebuf->buf);
		free(prebuf);
		return -1;
	}

	if (pthread_cond_init(&prebuf->data_avail_cond, NULL))
	{
		pthread_cond_destroy(&prebuf->space_avail_cond);
		pthread_mutex_destroy(&prebuf->lock);
		free(prebuf->buf);
		free(prebuf);
		return -1;
	}

	pthread_attr_init(&attr);
	if (pthread_create(&prebuf->thread, &attr, prebufmain, prebuf))
	{
		pthread_cond_destroy(&prebuf->data_avail_cond);
		pthread_cond_destroy(&prebuf->space_avail_cond);
		pthread_mutex_destroy(&prebuf->lock);
		free(prebuf->buf);
		free(prebuf);
		return -1;
	}

	pthread_mutex_lock(&prebuf_lock);
	prebuf->next = prebuf_hash[hashpos];
	prebuf_hash[hashpos] = prebuf;
	pthread_mutex_unlock(&prebuf_lock);

	return 0;
}


/*
 * Stops prebuffering a file.
 *
 * Input:
 *	Handle of prebuffered file.
 */
void prebufstop(int fd)
{
	int	hashpos = fd % HASH_SIZE;
	PREBUF	*pb = NULL, **pbptr;

	/*
	 * First remove the file from the hashtable so nobody else will
	 * be able to mess with it concurrently.
	 */
	pthread_mutex_lock(&prebuf_lock);
	for (pbptr = &prebuf_hash[hashpos]; &pbptr != NULL;
		pbptr = &(*pbptr)->next)
	{
		if ((*pbptr)->fd == fd)
		{
			pb = *pbptr;
			*pbptr = pb->next;
			break;
		}
	}
	pthread_mutex_unlock(&prebuf_lock);

	if (pb == NULL)
		return;

	/*
	 * Now dispose of the read thread.
	 */
	pthread_mutex_lock(&pb->lock);
	pb->interrupted = 1;
	pthread_mutex_unlock(&pb->lock);

	/* Send a signal to interrupt fread() or cond_wait() */
	if (! pb->done)
	{
		pthread_cond_signal(&pb->space_avail_cond);
		if (pthread_kill(pb->thread, SIGUSR1) == 0)
			pthread_join(pb->thread, NULL);
	}

	pthread_mutex_destroy(&pb->lock);
	pthread_cond_destroy(&pb->data_avail_cond);
	pthread_cond_destroy(&pb->space_avail_cond);
	free(pb->buf);
	free(pb);
}


/*
 * Copies bytes from a prebuffer to an application buffer.
 */
static int
copy_bytes(PREBUF *pb, char *buf, int len)
{
	int	bytes_avail, bytes_copied = 0;

	while (len > 0 && pb->head != pb->tail)
	{
		if (pb->head < pb->tail)
			bytes_avail = pb->tail - pb->head;
		else
			bytes_avail = pb->buf_size - pb->head;
		if (bytes_avail > len)
			bytes_avail = len;

		memcpy(buf, pb->buf + pb->head, bytes_avail);

		len -= bytes_avail;
		buf += bytes_avail;
		bytes_copied += bytes_avail;
		pb->head += bytes_avail;
		if (pb->head == pb->buf_size)
			pb->head = 0;
	}

	return bytes_copied;
}


/*
 * Reads from a prebuffered file.
 *
 * Input:
 *	Handle of prebuffered file.
 *	Buffer to fill.
 *	Size of buffer.
 * Returns:
 *	Number of bytes read, or -1 on failure.
 */
int prebufread(int fd, char *buf, int len)
{
	int	hashpos = fd % HASH_SIZE;
	int	bytes_copied, bytes_total = 0;
	PREBUF	*pb;

	pthread_mutex_lock(&prebuf_lock);
	for (pb = prebuf_hash[hashpos]; pb != NULL; pb = pb->next)
		if (pb->fd == fd)
			break;
	pthread_mutex_unlock(&prebuf_lock);
	if (pb == NULL)
	{
		errno = EBADF;
		return -1;
	}

	pthread_mutex_lock(&pb->lock);

	do {
		bytes_copied = copy_bytes(pb, buf, len);

		if (bytes_copied == 0)
		{
			if (pb->error)
			{
				pthread_mutex_unlock(&pb->lock);
				return -1;
			}
			if (pb->eof)
			{
				pthread_mutex_unlock(&pb->lock);
				return bytes_total;
			}
		}
		else
		{
			buf += bytes_copied;
			len -= bytes_copied;
			bytes_total += bytes_copied;
		}

		/* Let the read thread fill up the buffer some more. */
		pthread_cond_signal(&pb->space_avail_cond);

		/*
		 * If we want more bytes than were available, we'll have to
		 * block waiting for them.
		 */
		if (len > 0 && ! pb->done)
			pthread_cond_wait(&pb->data_avail_cond, &pb->lock);
	} while (len > 0);

	pthread_mutex_unlock(&pb->lock);
	return bytes_total;
}
