/*
 * Daemon to play M3Us in shuffle mode.
 *
 * Usage:
 *	m3ud {-stop|-pause|-play|-next|-prev|filename.m3u}
 *
 * -stop will kill the instance of m3ud currently in the background, if any.
 *
 * -pause will cause m3ud to pause whatever is currently playing.
 *
 * -play continues playing after a -pause; it has no effect otherwise.
 *
 * -next and -prev jump to the next and previous tracks respectively.
 *
 * If run with a pathname as argument, it's assumed to be an M3U.  The
 * current directory will be changed to the one containing the M3U so
 * that relative paths to MP3s will work.  Any existing M3U playback will
 * be stopped first.
 */ 
#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <string.h>
#include <signal.h>
#include <errno.h>

char command[100] = "";

static void
add_string_to_list(char ***list, int *count, char *string)
{
	char **newlist;

	if (*count == 0 || *list == NULL)
		newlist = malloc(sizeof(char *));
	else
		newlist = realloc(*list, sizeof(char *) * (*count + 1));
	if (newlist == NULL)
	{
		perror("malloc");
		exit(1);
	}

	*list = newlist;
	newlist[*count] = strdup(string);
	if (newlist[*count] == NULL)
	{
		perror("strdup");
		exit(1);
	}

	++*count;
}

// Picks a song and attempts to not repeat songs very often.
int
pick_song(int nsongs, int *history, int history_size)
{
	int	tries_left = nsongs;
	int	i;
	int	song;

	// Trivial case: Only one song!
	if (nsongs == 1)
		return 0;

	while (tries_left--)
	{
		song = random() % nsongs;
		for (i = 0; i < history_size; i++)
			if (song == history[i])
				break;
		if (i == history_size)
			break;
	}

	memmove(history + 1, history, sizeof(*history) * (history_size - 1));
	history[0] = song;

	return song;
}


/*
 * Signal handler.  Reads whatever command is waiting.
 */
void sighup(int dummy)
{
	FILE *fp;

	fp = fopen("/tmp/m3ud.command", "r");
	if (fp != NULL)
	{
		fgets(command, sizeof(command), fp);
		fclose(fp);
		strtok(command, "\n");

		unlink("/tmp/m3ud.command");
	}

	signal(SIGHUP, sighup);
}


/*
 * Sends a command to an existing m3ud.
 */
void send_command(char *cmd)
{
	FILE *fp;
	int pid;

	fp = fopen("/tmp/m3ud.pid", "r");
	if (fp == NULL)
		return;
	fscanf(fp, "%d", &pid);
	fclose(fp);

	fp = fopen("/tmp/m3ud.command", "w");
	if (fp == NULL)
	{
		perror("Can't write command file");
		return;
	}
	if (cmd[0] == '-')
		cmd++;
	fprintf(fp, "%s\n", cmd);
	fclose(fp);

	kill(pid, SIGHUP);
}


/*
 * Processes a command from another m3ud.  Signal handler is assumed to have
 * read the command file already.
 */
void process_command(int child_pid)
{
	FILE *fp;

	if (! strcmp(command, "stop"))
	{
		killpg(child_pid, SIGINT);
		fp = fopen("/tmp/m3ud.current-file", "w");
		if (fp != NULL)
		{
			fprintf(fp, "none\n");
			fclose(fp);
		}
		exit(0);
	}

	if (! strcmp(command, "pause"))
		killpg(child_pid, SIGSTOP);
	
	if (! strcmp(command, "play"))
		killpg(child_pid, SIGCONT);
	
	if (! strcmp(command, "next") || ! strcmp(command, "prev"))
	{
		killpg(child_pid, SIGINT);
		wait(NULL);
	}
}


main(int argc, char **argv)
{
	FILE	*fp;
	char	**paths = NULL;
	char	buf[1500];
	char	*c;
	int	nsongs = 0;
	int	songnum;
	int	child;
	int	*history;
	int	history_size;

	srandom(getpid());

	if (argc != 2)
	{
		fprintf(stderr, "Usage: %s -stop|-play|-pause|playlist.m3u\n",
			argv[0]);
		exit(1);
	}

	if (argv[1][0] == '-')
	{
		// Control an existing m3ud, if any.
		send_command(argv[1]);
		exit(0);
	}

	if (argv[1][0] == '/')
	{
		// Change to the M3U's directory.
		c = strrchr(argv[1], '/');
		if (c != NULL)	// Shouldn't be possible for it to be NULL!
		{
			*c = '\0';
			if (chdir(argv[1]))
			{
				perror(argv[1]);
				exit(1);
			}

			argv[1] = c + 1;
		}
	}

	fp = fopen(argv[1], "r");
	if (fp == NULL)
	{
		perror(argv[1]);
		exit(1);
	}

	while (fgets(buf, sizeof(buf), fp) != NULL)
	{
		if (buf[0] == '#' || buf[0] == ';')
			continue;

		for (c = buf; *c; c++)
		{
			if (*c == '\\')
				*c = '/';
			if (*c == '\r' || *c == '\n')
			{
				*c = '\0';
				break;
			}
		}

		if (buf[0] == '\0')
			continue;

		add_string_to_list(&paths, &nsongs, buf);
	}

	fclose(fp);

	// Stop an existing M3U if any.
	send_command("stop");

	// Become a daemon.
	switch (fork()) {
	case -1:
		perror("fork");
		exit(1);
	case 0:
		break;
	default:
		exit(0);
	}
	
	// Record our process ID so we can be signalled.
	fp = fopen("/tmp/m3ud.pid", "w");
	if (fp == NULL)
		perror("Can't record process ID!");
	else
	{
		fprintf(fp, "%d\n", getpid());
		fclose(fp);
	}

	// printf("Playlist has %d songs\n", nsongs);
	history_size = nsongs / 2;
	if (history_size > 0)
		history = malloc(history_size * sizeof(int));

	for (songnum = 0; songnum < history_size; songnum++)
		history[songnum] = -1;

	signal(SIGHUP, sighup);

	while (1)
	{
		songnum = pick_song(nsongs, history, history_size);
		// printf("Playing %s\n", paths[songnum]);
		// fflush(stdout);

		fp = fopen("/tmp/m3ud.current-file", "w");
		if (fp != NULL)
		{
			fprintf(fp, "%s\n", paths[songnum]);
			fclose(fp);
		}

		child = fork();
		if (child == 0)
		{
			setpgrp();
			execlp("madplay", "madplay", paths[songnum], NULL);
			perror("madplay");
			exit(1);
		}
		else if (child > 0)
		{
			while (1)
			{
				if (wait(NULL) < 0)
				{
					if (errno == EINTR)
					{
						process_command(child);
						if (! strcmp(command, "next"))
						{
							break;
						}
					}
					else
					{
						perror("wait");
					}
				}
				else
				{
					break;
				}
			}
		}
		else if (child < 0)
		{
			perror("fork");
			exit(1);
		}
	}
}
